[CmdletBinding(PositionalBinding = $false)]
param(
    [Parameter(Mandatory = $false)]
    [string]$targetMachine,
    
    [string]$targetMachinePublicname = $null,
    [string]$targetUsername = $null,
    [string]$frameworkLanguage = "dotnet",
    [bool]$updateTargetMachinePublicname = $false,
    [bool]$installApplications = $true,
    [bool]$installPyRunner = $true,
    [string]$installFilename = $null,
    [string]$executeDatabaseOps = $null,
    [string]$pyRunnerInstallOptions = $null,
    [bool]$getServiceReport = $false,
    [bool]$getProtocols = $false,
    [string]$getProtocolsFilter = $null,
    [bool]$updateProtocols = $false,
    [string]$updateProtocolsFilter = $null,
    [string]$updateProtocolsStatement = $null,
    [bool]$getDatabaseBackup = $false,
    [string]$restoreDatabaseFromBackupFilename = $null,
    [bool]$reinitializeInstrument = $false,
    [bool]$deleteAll = $false,
    [bool]$dryRun = $false,
    [switch]$help,
    [switch]$version
)

# Function to get script version from bodhi-apps.sh
function Get-ScriptVersion {
    $bashScriptPath = Join-Path $PSScriptRoot "bodhi-apps.sh"
    if (Test-Path $bashScriptPath) {
        $content = Get-Content $bashScriptPath
        foreach ($line in $content) {
            if ($line -match '^SCRIPT_VERSION="([^"]+)"') {
                return $matches[1]
            }
        }
    }
    return "Version not found"
}

# Function to get current ISO UTC datetime
function Get-IsoUtcDateTime {
    return [DateTime]::UtcNow.ToString("yyyy-MM-ddTHH:mm:ss.fffZ")
}

# Function to display help
function Show-Help {
    Write-Output ""
    Write-Output "============================================================================="
    Write-Output "                            BODHI APPS DEPLOYMENT SCRIPT"
    Write-Output "                                   version $scriptVersion"
    Write-Output "============================================================================="
    Write-Output ""
    Write-Output "DESCRIPTION:"
    Write-Output "This script installs and manages Bodhi applications and PyRunner on a Linux target machine."
    Write-Output "It requires PuTTY (plink.exe and pscp.exe) to be installed and available in the system PATH."
    Write-Output ""
    Write-Output "USAGE:"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine <hostname_or_ip> [additional_parameters]"
    Write-Output ""
    Write-Output "REQUIRED PARAMETERS:"
    Write-Output "  -targetMachine <string>                Target Linux machine hostname or IP address"
    Write-Output ""
    Write-Output "OPTIONAL PARAMETERS:"
    Write-Output "  -targetMachinePublicname <string>      Public name for the target machine (default: same as targetMachine)"
    Write-Output "  -targetUsername <string>               Username for SSH connection"
    Write-Output "  -frameworkLanguage <string>            Framework language: 'dotnet' or 'nodejs' (default: 'dotnet')"
    Write-Output "  -updateTargetMachinePublicname <bool>  Update the public name configuration (default: false)"
    Write-Output "  -installApplications <bool>            Install Bodhi applications (default: true)"
    Write-Output "  -installPyRunner <bool>                Install PyRunner (default: true)"
    Write-Output "  -installFilename <string>              Path to installation ZIP file (auto-detected if not specified)"
    Write-Output "  -executeDatabaseOps <string>           Database operations: nop | delete | recreate | reinitialize | migrate | seed | migrate+seed"
    Write-Output "  -pyRunnerInstallOptions <string>       Additional options for PyRunner installation"
    Write-Output "  -getServiceReport <bool>               Generate and download service report (default: false)"
    Write-Output "  -getProtocols <bool>                   Get protocols from database (default: false)"
    Write-Output "  -getProtocolsFilter <string>           SQL WHERE clause to filter protocols (default: empty)"
    Write-Output "  -updateProtocols <bool>                Update protocols in database (default: false)"
    Write-Output "  -updateProtocolsFilter <string>        SQL WHERE clause to filter protocols for update (default: empty)"
    Write-Output "  -updateProtocolsStatement <string>     SQL SET clause for protocol update (default: empty)"
    Write-Output "  -getDatabaseBackup <bool>              Create and download database backup (default: false)"
    Write-Output "  -restoreDatabaseFromBackupFilename <string>  Restore database from backup file"
    Write-Output "  -reinitializeInstrument <bool>         Reinitialize the instrument (default: false)"
    Write-Output "  -deleteAll <bool>                      Delete all Bodhi files and services (default: false)"
    Write-Output "  -dryRun <bool>                         Show what would be done without executing (default: false)"
    Write-Output "  -help                                  Show this help message"
    Write-Output "  -version                               Display the script version and exit"
    Write-Output ""
    Write-Output "EXAMPLES:"
    Write-Output "  # Install applications on a target machine"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2"
    Write-Output ""
    Write-Output "  # Install with specific install file and username"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine myserver.local -targetUsername operator -installFilename 'C:\builds\bodhi-apps.zip'"
    Write-Output ""
    Write-Output "  # Generate service report only"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -getServiceReport `$true"
    Write-Output ""
    Write-Output "  # Get all protocols from database"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -getProtocols `$true"
    Write-Output ""
    Write-Output "  # Get protocols from database, filter for the one protocol with name equal to 'MyProtocol' [ ! syntax is important !]"
    Write-Output '  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -getProtocols $true -getProtocolsFilter "name = ''MyProtocol''"'
    Write-Output ""
    Write-Output "  # Get protocols from database, filter for all protocols with description containing the word 'Revvity' [ ! syntax is important !]"
    Write-Output '  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -getProtocols $true -getProtocolsFilter "description LIKE ''%Revvity%''"'
    Write-Output ""
    Write-Output "  # Update protocols in database [ ! syntax is important !]"
    Write-Output '  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -updateProtocols $true -updateProtocolsFilter "name = ''MyProtocol'' AND is_current = true AND deleted_at IS NULL" -updateProtocolsStatement " is_current = false, last_modified_at = NOW(), last_modified_by = ''Revvity support''"'
    Write-Output ""
    Write-Output "  # Reinitialize the instrument (rehomes hardware and restarts software)"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -reinitializeInstrument `$true"
    Write-Output ""
    Write-Output "  # Display script version"
    Write-Output "  .\bodhi-apps.ps1 -version"
    Write-Output ""
    Write-Output "  # Dry run to see what would be executed"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -dryRun `$true"
    Write-Output ""
    Write-Output "  # Delete all Bodhi components (WARNING: destructive operation)"
    Write-Output "  .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -deleteAll `$true"
    Write-Output ""
    Write-Output "NOTES:"
    Write-Output "  - The target machine must be a Linux machine with unzip installed"
    Write-Output "  - PuTTY must be installed and available in the system PATH"
    Write-Output "  - The script will prompt for SSH credentials when executed"
    Write-Output "  - Some operations (deleteAll, restoreDatabaseFromBackup, updateProtocols) require confirmation"
    Write-Output "  - For updateProtocols, both updateProtocolsFilter and updateProtocolsStatement are required"
    Write-Output ""
    Write-Output "============================================================================="
    Write-Output ""
}

$scriptVersion = Get-ScriptVersion

# Check for help parameter first - if both help and version are specified, show help only
if ($help) {
    Show-Help
    exit
}

# Check for version parameter if help was not specified
if ($version) {
    Write-Output $scriptVersion
    exit
}

# Check for missing mandatory parameters (help and version already handled above)
if ([string]::IsNullOrEmpty($targetMachine)) {
    Show-Help
    exit
}

Write-Output ""
Write-Output "============================================================================="
Write-Output "Welcome to the Bodhi Apps Deployment Script, version $scriptVersion!"
Write-Output "This script will install and manage Bodhi applications on your target machine."
Write-Output "Target machine: $targetMachine"
Write-Output "============================================================================="
Write-Output ""

if ($dryRun) {
    Write-Output "============================================="
    Write-Output " This is a dry run, there will be no changes!"
    Write-Output "============================================="
    Write-Output ""
}

try {
    $plinkPath = (Get-Command plink.exe -ErrorAction SilentlyContinue).Source
    if ($null -eq $plinkPath) {
        throw "plink.exe not found"
    }
}
catch {
    Write-Output "putty plink.exe is not found in the system path, please install PuTTY"
    Write-Output "if you have installed PuTTY, please add the path to the system PATH variable and restart the shell"
    exit
}

try {
    $pscpPath = (Get-Command pscp.exe -ErrorAction SilentlyContinue).Source
    if ($null -eq $pscpPath) {
        throw "pscp.exe not found"
    }
}
catch {
    Write-Output "putty pscp.exe is not found in the system path, please install PuTTY"
    Write-Output "if you have installed PuTTY, please add the path to the system PATH variable and restart the shell"
    exit
}

if ( $pyRunnerInstallOptions -eq $null ) {
    $pyRunnerInstallOptions = ""
}

$doExecuteDatabaseOps = ( -not ( $executeDatabaseOps -eq $null -or $executeDatabaseOps -eq "" -or $executeDatabaseOps -eq "nop") )
$restoreDatabaseFromBackup = -not ($restoreDatabaseFromBackupFilename -eq $null -or $restoreDatabaseFromBackupFilename -eq "");

if ( $deleteAll ) {
    Write-Output "Remove the Bodhi postgres databases and all Bodhi and PyRunner files and services on the target machine!!"
    Write-Output "WARNING: this is an unrecoverable operation!!!!"
    Read-Host "Press Enter to continue or Ctrl+C to exit"

    # make sure we don't do anything else
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getProtocols = $false
    $updateProtocols = $false
    $getDatabaseBackup = $false
    $restoreDatabaseFromBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $getServiceReport ) {
    Write-Output "Will generate a Service Report on the target machine and download it here in the Downloads folder!!"
    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getProtocols = $false
    $updateProtocols = $false
    $getDatabaseBackup = $false
    $restoreDatabaseFromBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $getProtocols ) {
    Write-Output "Will get protocols from the database on the target machine and download the file here in the Downloads folder!!"
    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getDatabaseBackup = $false
    $restoreDatabaseFromBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $updateProtocols ) {
    if ([string]::IsNullOrEmpty($updateProtocolsFilter) -or [string]::IsNullOrEmpty($updateProtocolsStatement)) {
        Write-Output "ERROR: Both updateProtocolsFilter and updateProtocolsStatement are required for update operation"
        exit
    }
    Write-Output "Will update protocols in the database on the target machine!!"
    Write-Output "WARNING: this is an unrecoverable operation!!!!"
    
    Read-Host "Press Enter to continue or Ctrl+C to exit"

    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getProtocols = $false
    $getDatabaseBackup = $false
    $restoreDatabaseFromBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $getDatabaseBackup ) {
    Write-Output "Will generate a Database Backup and store it in the Downloads folder!!"
    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getProtocols = $false
    $updateProtocols = $false
    $restoreDatabaseFromBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $restoreDatabaseFromBackup ) {
    Write-Output "Will restore the database from a Database Backup file!!"
    Write-Output "WARNING: this is an unrecoverable operation!!!!"
    Read-Host "Press Enter to continue or Ctrl+C to exit"

    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getProtocols = $false
    $updateProtocols = $false
    $getDatabaseBackup = $false
    $executeDatabaseOps = "nop"
    $reinitializeInstrument = $false
}
elseif ( $reinitializeInstrument ) {
    Write-Output "Will reinitialze the instrument: rehome the hardware and restart the software!!"
    # make sure we don't do anything else
    $deleteAll = $false
    $updateTargetMachinePublicname = $false
    $installApplications = $false
    $installPyRunner = $false
    $getServiceReport = $false
    $getProtocols = $false
    $updateProtocols = $false
    $restoreDatabaseFromBackup = $false
    $getDatabaseBackup = $false
    $executeDatabaseOps = "nop"
}
elseif ( $doExecuteDatabaseOps ) {
    if (( -not $installApplications ) -and ( -not $installPyRunner)) {
        Write-Output "Will $executeDatabaseOps the databases on the target machine"
        Write-Output "WARNING: this is an unrecoverable operation!!!!"
        Read-Host "Press Enter to continue or Ctrl+C to exit"
    }
}
else {
    if ( ( $frameworkLanguage -eq 'dotnet' ) -or ( $frameworkLanguage -eq 'nodejs')) {
        if ( $frameworkLanguage -eq 'nodejs') {
            Write-Output "================================="
            Write-Output "framework: $frameworkLanguage"
            Write-Output "================================="
            Write-Output ""
        }
    }
    else {
        Write-Output "ERROR: wrong frameworkLanguage=${frameworkLanguage}, only 'dotnet' and 'nodejs' are supported, exit now"
        exit
    }

    if ( $installApplications -or $installPyRunner ) {

        # make sure we don't do anything else
        $updateTargetMachinePublicname = $false
        $deleteAll = $false
        $getServiceReport = $false
        $getProtocols = $false
        $updateProtocols = $false
        $getDatabaseBackup = $false
        $restoreDatabaseFromBackup = $false
        $reinitializeInstrument = $false

        # an install file is requiered 
        # Check if the install file exists
        if ($installFilename -eq $null -or $installFilename -eq "") {
            # If no install file is provided, we will use the first one in the current directory
            $installFilename = Get-ChildItem -Path .\ -Filter *.zip | Sort-Object -Property @{ Expression = 'Name'; Ascending = $false } | Select-Object -First 1
            if ($null -eq $installFilename) {
                Write-Output "No install file found in the current directory"
                exit
            }   
            Write-Output "No install file provided. Using $installFilename found in the current directory"
            Write-Output "If you want to use a different install file, please provide the full path to the install file using the -installFileName parameter"

            Write-Output "Confirm the install file: ${installFilename}"
            Read-Host "Press Enter to continue or Ctrl+C to exit"
        }
    }
}

if ( $installApplications -or $updateTargetMachinePublicname ) {
    if ($targetMachinePublicname -eq $null -or $targetMachinePublicname -eq "") {
        # same as targetMachine
        $targetMachinePublicname = $targetMachine
    }
}

if ( $installApplications -or $installPyRunner -or $updateTargetMachinePublicname -or $deleteAll -or $getServiceReport -or $getProtocols -or $updateProtocols -or $getDatabaseBackup -or $restoreDatabaseFromBackup -or $doExecuteDatabaseOps -or $reinitializeInstrument) {
    # get credentials
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] require to enter the credentials for the target machine: '${targetMachine}'"
    }
    else {
        $credential = Get-Credential -Message "Enter the credentials for the target machine: '${targetMachine}'" -UserName $targetUsername
    
        if ($null -eq $credential) {
            Write-Output "No credentials provided. Exiting"
            exit
        }
    
        $targetUsername = $credential.UserName
    
        $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($credential.Password)
        $targetPassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
    }
}

# Print all parameters for debugging
Write-Output ""
Write-Output "============================================="
Write-Output "Script Parameters:"
Write-Output "============================================="
Write-Output "targetMachine: $targetMachine"
Write-Output "targetMachinePublicname: $targetMachinePublicname"
Write-Output "targetUsername: $targetUsername"
Write-Output "frameworkLanguage: $frameworkLanguage"
Write-Output "updateTargetMachinePublicname: $updateTargetMachinePublicname"
Write-Output "installApplications: $installApplications"
Write-Output "installPyRunner: $installPyRunner"
Write-Output "installFilename: '$installFilename'"
Write-Output "executeDatabaseOps: '$executeDatabaseOps'"
Write-Output "pyRunnerInstallOptions: '$pyRunnerInstallOptions'"
Write-Output "getServiceReport: $getServiceReport"
Write-Output "getProtocols: $getProtocols"
Write-Output "getProtocolsFilter: '$getProtocolsFilter'"
Write-Output "updateProtocols: $updateProtocols"
Write-Output "updateProtocolsFilter: '$updateProtocolsFilter'"
Write-Output "updateProtocolsStatement: '$updateProtocolsStatement'"
Write-Output "getDatabaseBackup: $getDatabaseBackup"
Write-Output "restoreDatabaseFromBackupFilename: '$restoreDatabaseFromBackupFilename'"
Write-Output "reinitializeInstrument: $reinitializeInstrument"
Write-Output "deleteAll: $deleteAll"
Write-Output "dryRun: $dryRun"
Write-Output "============================================="
Write-Output ""


if ( $installApplications -or $installPyRunner -or $updateTargetMachinePublicname -or $deleteAll -or $getServiceReport -or $getProtocols -or $updateProtocols -or $getDatabaseBackup -or $restoreDatabaseFromBackup -or $doExecuteDatabaseOps -or $reinitializeInstrument) {
    # Copy the artifacts to target machine
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] copy files to the target machine "
    }
    else {
        Write-Output "copying the files to the target machine..."

        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "echo ${targetPassword} | sudo -S rm -r -f /tmp/bodhi && mkdir -p '/tmp/bodhi/software-updates/apps' '/tmp/bodhi/software-updates/scripts' && echo ''"

        if ( $installApplications -or $installPyRunner ) {

            # an install file is requiered 
            # full path to the install file
            $installFile = Get-Item -Path $installFilename
            Write-Output "Using install file: $($installFile.FullName)"
            pscp.exe  -pw $targetPassword $($installFile.FullName) "${targetUsername}@${targetMachine}:/tmp/bodhi/software-updates/apps/$($installFile.Name)"
        }
        pscp.exe -pw $targetPassword "bodhi-apps.ps1" "${targetUsername}@${targetMachine}:/tmp/bodhi/software-updates/scripts/bodhi-apps.ps1"
        pscp.exe -pw $targetPassword "bodhi-apps.sh" "${targetUsername}@${targetMachine}:/tmp/bodhi/software-updates/scripts/bodhi-apps.sh"
        pscp.exe -pw $targetPassword -r "./lib" "${targetUsername}@${targetMachine}:/tmp/bodhi/software-updates/scripts"
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "echo ${targetPassword} | sudo -S chmod +x /tmp/bodhi/software-updates/scripts/bodhi-apps.sh && echo ${targetPassword} | sudo -S find /tmp/bodhi/software-updates/scripts/lib -name '*.sh' -exec chmod +x {} \;&& echo ''"
        Write-Output ""
    }
}

if ( $deleteAll ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] remove all bodhi files and services on the target machine..."
    }
    else {
        Write-Output "remove the bodhi postgres database and all bodhi files and services on the target machine..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --delete-all yes --client-datetime '$clientDateTime'"
    }

    exit
}

if ( $getServiceReport ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] generate a Service Report on the target machine and download it here..."
    }
    else {
        $tempPath = [System.IO.Path]::GetTempPath()
        $tempFileName = [System.IO.Path]::Combine($tempPath, [System.IO.Path]::GetRandomFileName())

        # generate the Service Report on the target machine 
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --generate-service-report yes --client-datetime '$clientDateTime'" > "${tempFileName}"

        $fileContent = Get-Content -Path "${tempFileName}"
        Remove-Item -Path "${tempFileName}"

        $exitStatus = $fileContent | Select-String -Pattern "GENERATE_SERVICE_REPORT_EXIT_STATUS" | ForEach-Object {
            $line = $_ -split ' '
            $line[1]
        }
        if ( $exitStatus -eq "true" ) { 
            
            $serviceReportName = $fileContent | Select-String -Pattern "SERVICE_REPORT_NAME" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }

            $serviceReportZipFileName = $fileContent | Select-String -Pattern "SERVICE_REPORT_ZIP_FILE_NAME" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }
        
            $serviceReportZipFilePath = $fileContent | Select-String -Pattern "SERVICE_REPORT_ZIP_FILE_PATH" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }

            $shell = New-Object -ComObject Shell.Application
            $downloadsFolder = $shell.Namespace('shell:Downloads').Self.Path
            $downloadedZipFilePath = Join-Path -Path $downloadsFolder -ChildPath $serviceReportZipFileName

            # get the file from target machine 
            pscp.exe -pw $targetPassword "${targetUsername}@${targetMachine}:${serviceReportZipFilePath}" "${downloadedZipFilePath}"

            # delete the file on target machine 
            plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "echo ${targetPassword} | sudo -S rm -f '$serviceReportZipFilePath'"

            Write-Output ""
            Write-Output "Please find the downloaded Service Report at ${downloadedZipFilePath}"
            Write-Output ""
            Write-Output ""
        }
        else {
            $exitError = $fileContent | Select-String -Pattern "GENERATE_SERVICE_REPORT_EXIT_ERROR" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }
    
            Write-Output ""
            Write-Output "ERROR: failed to generate a Service Report, $exitError"
            Write-Output ""
        }
    }

    exit
}

if ( $getProtocols ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] get protocols from database on the target machine and download it here..."
    }
    else {
        $tempPath = [System.IO.Path]::GetTempPath()
        $tempFileName = [System.IO.Path]::Combine($tempPath, [System.IO.Path]::GetRandomFileName())

        # get protocols from database on the target machine 
        $clientDateTime = Get-IsoUtcDateTime
        if ([string]::IsNullOrEmpty($getProtocolsFilter)) {
            plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --dump-protocols yes --client-datetime '$clientDateTime'" > "${tempFileName}"
        }
        else {
            $escapedFilter = $getProtocolsFilter -replace "'", "'\''"
            plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --dump-protocols yes --dump-protocols-filter '${escapedFilter}' --client-datetime '$clientDateTime'" > "${tempFileName}"
        }

        $fileContent = Get-Content -Path "${tempFileName}"
        Remove-Item -Path "${tempFileName}"

        $exitStatus = $fileContent | Select-String -Pattern "DUMP_PROTOCOLS_EXIT_STATUS" | ForEach-Object {
            $line = $_ -split ' '
            $line[1]
        }
        if ( $exitStatus -eq "true" ) { 
            
            $protocolsFileName = $fileContent | Select-String -Pattern "DUMP_PROTOCOLS_FILE_NAME" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }

            $protocolsFilePath = $fileContent | Select-String -Pattern "DUMP_PROTOCOLS_FILE_PATH" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }

            $shell = New-Object -ComObject Shell.Application
            $downloadsFolder = $shell.Namespace('shell:Downloads').Self.Path
            $downloadedFilePath = Join-Path -Path $downloadsFolder -ChildPath $protocolsFileName

            # get the file from target machine 
            pscp.exe -pw $targetPassword "${targetUsername}@${targetMachine}:${protocolsFilePath}" "${downloadedFilePath}"

            # delete the file on target machine 
            plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "echo ${targetPassword} | sudo -S rm -f '$protocolsFilePath'"

            Write-Output ""
            Write-Output "Please find the downloaded Protocols file at ${downloadedFilePath}"
            Write-Output ""
            Write-Output ""
        }
        else {
            $exitError = $fileContent | Select-String -Pattern "DUMP_PROTOCOLS_EXIT_ERROR" | ForEach-Object {
                $line = $_ -split ' '
                $line[1]
            }
    
            Write-Output ""
            Write-Output "ERROR: failed to get protocols from database, $exitError"
            Write-Output ""
        }
    }

    exit
}

if ( $updateProtocols ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] update protocols in database on the target machine..."
    }
    else {
        Write-Output "Update protocols in database on the target machine..."
        $clientDateTime = Get-IsoUtcDateTime
        $escapedFilter = $updateProtocolsFilter -replace "'", "'\''"
        $escapedStatement = $updateProtocolsStatement -replace "'", "'\''"
        
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --update-protocols yes --update-protocols-filter '${escapedFilter}' --update-protocols-statement '${escapedStatement}' --client-datetime '$clientDateTime'"
    }

    exit
}

if ( $getDatabaseBackup ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] generate a Database Backup on the target machine and download it here..."
    }
    else {
        $tempPath = [System.IO.Path]::GetTempPath()
        $uniqueDirName = [System.IO.Path]::Combine($tempPath, [System.Guid]::NewGuid().ToString())
        New-Item -Path "${uniqueDirName}" -ItemType "directory" | out-null;
        Write-Output ""

        # stop all bodhi services
        Write-Output ""
        Write-Output "Stopping all Bodhi services..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --stop-services yes --client-datetime '$clientDateTime'"
        Write-Output "done"
        Write-Output "------------------------------"
        Write-Output ""
        Write-Output ""

        $backupBodhiStroutFileName = [System.IO.Path]::Combine($tempPath, $uniqueDirName, "backup.bodhi.strout.txt")

        # generate the Database Backup and get it
        Write-Output "Backup the Postgres bodhi databases, please wait..."
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "rm -rf /tmp/bodhi/backup/db && rm -f /tmp/bodhi/backup/db.zip && `
            mkdir -p /tmp/bodhi/backup/db && chmod 777 /tmp/bodhi/backup/ && chmod 777 /tmp/bodhi/backup/db && cd /tmp/bodhi/backup && `
            sudo -u postgres pg_dump -v --dbname 'bodhi' --clean --create --if-exists --no-password --inserts --quote-all-identifiers --format 'c' -f -b -f ./db/db_bodhi.custom && `
            sudo -u postgres pg_dump -v --dbname 'bodhi-idp' --clean --create --if-exists --no-password --inserts --quote-all-identifiers --format 'c' -f -b -f ./db/db_bodhi-idp.custom && `
            zip -rj db.zip db" > "${backupBodhiStroutFileName}"
        pscp.exe -pw $targetPassword "${targetUsername}@${targetMachine}:/tmp/bodhi/backup/db.zip" "db.zip"
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "rm -rf /tmp/bodhi/backup/db && rm -f /tmp/bodhi/backup/db.zip"

        Write-Output "done"
        Write-Output "---------------------------------------------------"
        Write-Output ""
        Write-Output ""

        # start all bodhi services
        Write-Output "Starting all Bodhi services..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --start-services yes --client-datetime '$clientDateTime'"
        Write-Output "done"
        Write-Output "------------------------------"
        Write-Output ""
        Write-Output ""

        if ( Test-Path -Path "$backupBodhiStroutFileName" ) {
            # move the zip to Downloads folder
            Write-Output ""
            $timestamp = Get-Date -Format "yyyy-MM-dd_HH-mm-ss"
            $databaseBackupZipFileName = "db_backup_${timestamp}.zip"

            $shell = New-Object -ComObject Shell.Application
            $downloadsFolder = $shell.Namespace('shell:Downloads').Self.Path
            $downloadedZipFilePath = Join-Path -Path $downloadsFolder -ChildPath $databaseBackupZipFileName

            Rename-Item -Path "db.zip" -NewName "$databaseBackupZipFileName"  -Force
            Move-Item -Path "$databaseBackupZipFileName" -Destination "$downloadedZipFilePath" -Force

            Write-Output "done"
            Write-Output "------------------------------------"
            Write-Output ""
            Write-Output ""

            Write-Output ""
            Write-Output "Please find the generated Database Backup at ${downloadedZipFilePath}"
            Write-Output ""
            Write-Output ""
        }
        else {
            Write-Output ""
            Write-Output "ERROR: failed to generate a Database Backup"
            Write-Output ""
        }

        Remove-Item -Path "${uniqueDirName}" -Recurse -Force
    }

    exit
}

if ( $restoreDatabaseFromBackup ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] restore the database from a Database Backup on the target machine..."
    }
    else {
        $tempPath = [System.IO.Path]::GetTempPath()
        $uniqueDirName = [System.IO.Path]::Combine($tempPath, [System.Guid]::NewGuid().ToString())
        New-Item -Path "${uniqueDirName}" -ItemType "directory" | out-null;

        $restoreBodhiStroutFileName = [System.IO.Path]::Combine($tempPath, $uniqueDirName, "restore.bodhi.strout.txt")

        # stop all bodhi services
        Write-Output "Stopping all Bodhi services..."
        Write-Output "------------------------------"
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --stop-services yes --client-datetime '$clientDateTime'"
        Write-Output ""
        Write-Output "------------------------------"
        Write-Output ""
        Write-Output ""

        # recreate the databases: this will delete the old databases and create new empty ones
        Write-Output "Recreate the bodhi databases, please wait..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --install-applications no --execute-database-ops recreate --client-datetime '$clientDateTime'"
        Write-Output "done"
        Write-Output "-------------------------------------------"
        Write-Output ""
        Write-Output ""

        # restore the Database Backup on the target machine using the backup files from the zip file
        Write-Output "Restore the bodhi databases, please wait..."
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "rm -rf /tmp/bodhi/backup/db && rm -f /tmp/bodhi/backup/db.zip && mkdir -p /tmp/bodhi/backup/db && `
            chmod 777 /tmp/bodhi/backup/ && chmod 777 /tmp/bodhi/backup/db"
        pscp.exe  -pw $targetPassword $("$restoreDatabaseFromBackupFilename") "${targetUsername}@${targetMachine}:/tmp/bodhi/backup/db.zip"
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/backup && `
            unzip db.zip -d ./db && `
            sudo -u postgres pg_restore -j 4 -v --dbname 'bodhi' --format 'c' ./db/db_bodhi.custom && `
            sudo -u postgres pg_restore -j 4 -v --dbname 'bodhi-idp' --format 'c' ./db/db_bodhi-idp.custom && `
            rm -rf /tmp/bodhi/backup/db && rm -f /tmp/bodhi/backup/db.zip" > $restoreBodhiStroutFileName
        Write-Output "done"
        Write-Output "-------------------------------------------"
        Write-Output ""
        Write-Output ""

        # start all bodhi services
        Write-Output "Starting all Bodhi services..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --start-services yes --client-datetime '$clientDateTime'"
        Write-Output "done"
        Write-Output "------------------------------"
        Write-Output ""
        Write-Output ""

        if ( Test-Path -Path "${restoreBodhiStroutFileName}" ) {
            Write-Output ""
            Write-Output "Succeeded to restore the databases from the Database Backup ${restoreDatabaseFromBackupFilename}"
            Write-Output ""
            Write-Output ""
        }
        else {
            Write-Output ""
            Write-Output "ERROR: failed to restore the databases from the Database Backup ${restoreDatabaseFromBackupFilename}"
            Write-Output ""
        }

        Remove-Item -Path "${uniqueDirName}" -Recurse -Force
    }

    exit
}


if ( $installApplications -or $installPyRunner ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] install, configure and run the application on the target machine..."
    }
    else {
        if ( $executeDatabaseOps -eq $null -or $executeDatabaseOps -eq "" ) {
            # when we install the apps, if not specified
            if ( $frameworkLanguage -eq 'dotnet' ) {
                $executeDatabaseOps = "migrate+seed"
            }
            if ( $frameworkLanguage -eq 'nodejs') {
                $executeDatabaseOps = "nop"
            }
        }

        $installApplicationsStr = "no"
        if ( $installApplications ) {
            $installApplicationsStr = "yes"
        }

        $installPyRunnerStr = "no"
        if ( $installPyRunner ) {
            $installPyRunnerStr = "yes"

            # always set the DEVICEIP=IpAddress in pyRunnerInstallOptions
            if ( $pyRunnerInstallOptions -eq $null -or $pyRunnerInstallOptions -eq "" -or $pyRunnerInstallOptions -notmatch 'DEVICEIP=' ) {
                $DEVICEIP = $null
                # check targetMachine if is an Ip Address
                if ( $targetMachine -match '^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$' ) {
                    $DEVICEIP = $targetMachine
                }
                else {
                    # resolve the targetMachine to an Ip Address
                    try {   
                        $ipAddress = [System.Net.Dns]::GetHostAddresses($targetMachine) | Where-Object { $_.AddressFamily -eq 'InterNetwork' }
                        if ($ipAddress) {
                            $DEVICEIP = $ipAddress[0].ToString()
                        }
                        else {
                            Write-Output "ERROR: could not resolve the target machine to an IP address"
                        }
                    }
                    catch {
                        Write-Output "ERROR: could not resolve the target machine to an IP address"
                    }
                }
                if ( $null -eq $DEVICEIP ) {
                    Write-Output "set DEVICEIP to default 192.168.0.2 !"
                    $DEVICEIP = "192.168.0.2"
                }

                if ( $null -ne $DEVICEIP ) {
                    Write-Output "DEVICEIP=$DEVICEIP"
                    if ($pyRunnerInstallOptions -ne $null -and $pyRunnerInstallOptions -ne "") {
                        $pyRunnerInstallOptions = "$pyRunnerInstallOptions DEVICEIP=$DEVICEIP"
                    }
                    else {
                        $pyRunnerInstallOptions = "DEVICEIP=$DEVICEIP"
                    }
                }
                else {
                    Write-Output "DEVICEIP will not be set !"
                }
            }

            # if not specified, always install PyRunner even if already installed
            if ( $pyRunnerInstallOptions -eq $null -or $pyRunnerInstallOptions -eq "" -or $pyRunnerInstallOptions -notmatch 'OVERWRITE=' ) {
                if ($pyRunnerInstallOptions -ne $null -and $pyRunnerInstallOptions -ne "") {
                    $pyRunnerInstallOptions = "$pyRunnerInstallOptions OVERWRITE=true"
                }
                else {
                    $pyRunnerInstallOptions = "OVERWRITE=true"
                }
            }

        }
        else {
            # if we don't install pyrunner, we don't need to pass the options
            $pyRunnerInstallOptions = ""
        }

        Write-Output "Run bodhi-apps.sh on the target machine to install, configure and run the bodhi applications..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --create-services overwrite --install-applications ${installApplicationsStr} --install-pyrunner ${installPyRunnerStr} --install-applications-filename '$($installFile.Name)' --host-public-name $targetMachinePublicname --framework-language ${frameworkLanguage} --set-localization yes --execute-database-ops ${executeDatabaseOps} --pyrunner-install-options '${pyRunnerInstallOptions}' --client-datetime '$clientDateTime'"
    }

    exit
}

if ( $updateTargetMachinePublicname ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] set the configuration setting 'application access public name' on the target machine..."
    }
    else {
        Write-Output "Run bodhi-apps.sh on the target machine to set 'application access public name' on the target machine..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --install-applications no --update-host-public-name yes --host-public-name $targetMachinePublicname --framework-language ${frameworkLanguage} --client-datetime '$clientDateTime'"
    }

    exit
}

if ( $doExecuteDatabaseOps ) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] execute these database operations: ${executeDatabaseOps}..."
    }
    else {
        Write-Output "Execute these database operations: ${executeDatabaseOps}..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --framework-language ${frameworkLanguage} --set-localization yes --execute-database-ops ${executeDatabaseOps} --client-datetime '$clientDateTime'"
    }

    exit
}

if ($reinitializeInstrument) {
    if ($dryRun) {
        Write-Output "Dry run: will [ but NOT now ] reinitialize the instrument on the target machine..."
    }
    else {
        Write-Output "Reinitialize the instrument on the target machine..."
        $clientDateTime = Get-IsoUtcDateTime
        plink.exe -ssh "${targetUsername}@${targetMachine}" -pw $targetPassword -batch "cd /tmp/bodhi/software-updates/scripts && echo ${targetPassword} | sudo -S USER_NAME=${targetUsername} ./bodhi-apps.sh --reinitialize-instrument yes --client-datetime '$clientDateTime'"
    }

    exit
}

Write-Output "Nothing todo, exit now"


