#!/usr/bin/env bash

set +e  # Disable exit on error

# Scripts have their own versioning, this is the version of the scripts ONLY
# This version is used to track changes in the scripts themselves.
# and is not related to the application version.
# that's because the scripts have their own lifecycle and versioning.
# DEVELOPERS: update this version when you make changes to the scripts.
# ALWAYS follow the versioning scheme: major.minor.patch
# - major: changes that break backward compatibility
# - minor: new features that are backward compatible
# - patch: bug fixes or minor changes
# NOTE: we should never break backward compatibility in the scripts
SCRIPT_VERSION="1.2.9"

main() {

  SERVER_TIMESTAMP=$(get_server_timestamp)

  # Check for version parameter first, before processing anything else
  for arg in "$@"; do
    if [[ "$arg" == "--version" ]]; then
      echo "$SCRIPT_VERSION"
      exit 0
    fi
  done

  # Default values
  FRAMEWORK_LANGUAGE="dotnet"
  INSTALL_APPLICATIONS="no"
  INSTALL_PYRUNNER="undefined"
  INSTALL_APPLICATIONS_FILENAME=""
  HOST_PUBLIC_NAME=""
  UPDATE_HOST_PUBLIC_NAME="no"
  CREATE_SERVICES="no"
  STOP_SERVICES="no"
  START_SERVICES="no"
  RESTART_SERVICES="no"
  STOP_APPS_PROCS="no"
  START_APPS_PROCS="no"
  STOP_CONTROL_PROCS="no"
  START_CONTROL_PROCS="no"
  EXECUTE_DATABASE_OPS="nop"
  LOCALE_LANG="en_US.UTF-8"
  TIME_ZONE="UTC"
  SET_LOCALIZATION="no"
  GENERATE_SERVICE_REPORT="no"
  DUMP_PROTOCOLS="no"
  DUMP_PROTOCOLS_FILTER=""
  UPDATE_PROTOCOLS="no"
  UPDATE_PROTOCOLS_FILTER=""
  UPDATE_PROTOCOLS_STATEMENT=""
  DELETE_ALL="no"
  REINITIALIZE_INSTRUMENT="no"
  PYRUNNER_INSTALL_OPTIONS=""
  SESSION_ID=""
  MASTER_SESSION_ID=""
  CLIENT_DATETIME=""
  # Loop through all the positional parameters
  while (( "$#" )); do
    case "$1" in
      --session-id)
        SESSION_ID="$2"
        shift 2
        ;;
      --master-session-id)
        MASTER_SESSION_ID="$2"
        shift 2
        ;;
      --client-datetime)
        CLIENT_DATETIME="$2"
        shift 2
        ;;
      --help)
        usage
        exit 0
        ;;
      --framework-language)
        FRAMEWORK_LANGUAGE="$2"
        shift 2
        ;;
      --install-applications)
        INSTALL_APPLICATIONS="$2"
        shift 2
        ;;
      --install-pyrunner)
        INSTALL_PYRUNNER="$2"
        shift 2
        ;;
      --install-applications-filename)
        INSTALL_APPLICATIONS_FILENAME="$2"
        shift 2
        ;;
      --host-public-name)
        HOST_PUBLIC_NAME="$2"
        shift 2
        ;;
      --update-host-public-name)
        UPDATE_HOST_PUBLIC_NAME="$2"
        shift 2
        ;;
      --create-services)
        CREATE_SERVICES="$2"
        shift 2
        ;;
      --stop-services)
        STOP_SERVICES="$2"
        shift 2
        ;;
      --start-services)
        START_SERVICES="$2"
        shift 2
        ;;
      --restart-services)
        RESTART_SERVICES="$2"
        shift 2
        ;;
      --stop-apps-processes)
        STOP_APPS_PROCS="$2"
        shift 2
        ;;
      --start-apps-processes)
        START_APPS_PROCS="$2"
        shift 2
        ;;
      --stop-control-processes)
        STOP_CONTROL_PROCS="$2"
        shift 2
        ;;
      --start-control-processes)
        START_CONTROL_PROCS="$2"
        shift 2
        ;;
      --execute-database-ops)
        EXECUTE_DATABASE_OPS="$2"
        shift 2
        ;;
      --locale-language)
        LOCALE_LANG="$2"
        shift 2
        ;;
      --time-zone)
        TIME_ZONE="$2"
        shift 2
        ;;
      --set-localization)
        SET_LOCALIZATION="$2"
        shift 2
        ;;
      --generate-service-report)
        GENERATE_SERVICE_REPORT="$2"
        shift 2
        ;;
      --dump-protocols)
        DUMP_PROTOCOLS="$2"
        shift 2
        ;;
      --dump-protocols-filter)
        DUMP_PROTOCOLS_FILTER="$2"
        shift 2
        ;;
      --update-protocols)
        UPDATE_PROTOCOLS="$2"
        shift 2
        ;;
      --update-protocols-filter)
        UPDATE_PROTOCOLS_FILTER="$2"
        shift 2
        ;;
      --update-protocols-statement)
        UPDATE_PROTOCOLS_STATEMENT="$2"
        shift 2
        ;;
      --delete-all)
        DELETE_ALL="$2"
        shift 2
        ;;
      --reinitialize-instrument)
        REINITIALIZE_INSTRUMENT="$2"
        shift 2
        ;;
      --pyrunner-install-options)
        PYRUNNER_INSTALL_OPTIONS="$2"
        shift 2
        ;;
      --) # end argument parsing
        shift
        break
        ;;
      -*|--*=) # unsupported flags
        echo "Warning: Ignoring unsupported flag $1" >&2
        shift
        ;;
      *) # preserve positional arguments
        PARAMS="$PARAMS $1"
        shift
        ;;
    esac
  done

  # set positional arguments in their proper place
  eval set -- "$PARAMS"

  if [[ "${USER_NAME}" == "" ]]; then
    USER_NAME=$(whoami)
  fi

  if [[ "$INSTALL_PYRUNNER" == "undefined" ]]; then
    # --install-pyrunner not passed as param, set it the same as --install-applications, for backward compatibility
    if [[ "$INSTALL_APPLICATIONS" == "yes" ]]; then
      INSTALL_PYRUNNER="yes"
    else
      INSTALL_PYRUNNER="no"
    fi
  fi

  run_datetime=$(date +"%Y-%m-%d_%H-%M-%S-%3N")
 
  if [[ "$DELETE_ALL" == "yes" ]]; then
    # no logging for delete_all, it will delete the log file as well
    echo ""
  else

    # Setup logging - capture all output to both stdout and log file
    run_scripts_log_directory="${bodhi_folder_run_scripts_reports}/${run_datetime}"
    run_scripts_log_file="${run_scripts_log_directory}/run.log"
    mkdir -p "$run_scripts_log_directory"

    # Use exec to redirect all subsequent output to both stdout and log file
    exec > >(tee -a "$run_scripts_log_file")
    exec 2> >(tee -a "$run_scripts_log_file" >&2)

    echo ""
    echo "======================================================================="
    echo "logging to: $run_scripts_log_file ..."
    echo "======================================================================="
  fi

  echo ""
  echo "This script is: $0 running on host: '$(hostname)/$(get_ip_address)' for user: '${USER_NAME}'"
  echo "Script version: $SCRIPT_VERSION"
  echo "Script parameters:"
  echo "  Client DateTime:              '$CLIENT_DATETIME'"
  echo "  Framework:                    $FRAMEWORK_LANGUAGE"
  echo "  Install Applications:         $INSTALL_APPLICATIONS"
  echo "  Install PyRunner:             $INSTALL_PYRUNNER [if part of the install file]"
  echo "  Install Applications Filename:'$INSTALL_APPLICATIONS_FILENAME'"
  echo "  Host public name:             '$HOST_PUBLIC_NAME'"
  echo "  Update host public name       $UPDATE_HOST_PUBLIC_NAME"
  echo "  Create All Services:          $CREATE_SERVICES"
  echo "  Stop All Services:            $STOP_SERVICES"
  echo "  Start All Services:           $START_SERVICES"
  echo "  Restart All Services:         $RESTART_SERVICES"
  echo "  Stop bodhi-apps processes:    $STOP_APPS_PROCS"
  echo "  Start bodhi-apps processes:   $START_APPS_PROCS"
  echo "  Stop bodhi-control processes: $STOP_CONTROL_PROCS"
  echo "  Start bodhi-control processes:$START_CONTROL_PROCS"
  echo "  Execute database operations:  '$EXECUTE_DATABASE_OPS'"
  echo "  PyRunner install options:     '$PYRUNNER_INSTALL_OPTIONS'"
  echo "  Locale language:              $LOCALE_LANG"
  echo "  Timezone:                     $TIME_ZONE"
  echo "  Set localization:             $SET_LOCALIZATION"
  echo "  Generate Service Report:      $GENERATE_SERVICE_REPORT"
  echo "  Dump Protocols:               $DUMP_PROTOCOLS"
  echo "  Dump Protocols Filter:        '$DUMP_PROTOCOLS_FILTER'"
  echo "  Update Protocols:             $UPDATE_PROTOCOLS"
  echo "  Update Protocols Filter:      '$UPDATE_PROTOCOLS_FILTER'"
  echo "  Update Protocols Statement:   '$UPDATE_PROTOCOLS_STATEMENT'"
  echo "  Reinitialize Instrument:      $REINITIALIZE_INSTRUMENT"
  echo "  Remove bodhi:                 $DELETE_ALL"
  echo "  Session ID:                   $SESSION_ID"
  echo "  Master Session ID:            $MASTER_SESSION_ID"
  echo ""


  if [[ -n "$CLIENT_DATETIME" ]]; then
    echo "--------------------------------------------------------"
    check_for_datetime_skew "$CLIENT_DATETIME" "$SERVER_TIMESTAMP"
    echo "--------------------------------------------------------"
    echo ""
  fi

  if [[ "$DELETE_ALL" == "yes" ]]; then
    delete_all
    exit 0 # yes, we exit the script here, delete_all will call all the other functions on its own
  fi
  
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]] || [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    echo "==================================="
    echo "framework: $FRAMEWORK_LANGUAGE"
    echo "==================================="
    echo ""
  else
    echo "================================================================================================"
    echo "wrong framework=$FRAMEWORK_LANGUAGE, only 'dotnet' and 'nodejs' are supported, exit now"
    echo "================================================================================================"
    exit 1
  fi

  if [[ "$GENERATE_SERVICE_REPORT" == "yes" ]]; then
    generate_service_report
    
    echo ""
    echo "see log at: $run_scripts_log_file"
    echo ""

    exit 0 # yes, we exit the script here, generate_service_report will call all the other functions on its own
  fi

  if [[ "$DUMP_PROTOCOLS" == "yes" ]]; then
    dump_protocols_from_database "$DUMP_PROTOCOLS_FILTER"

    echo ""
    echo "see log at: $run_scripts_log_file"
    echo ""

    exit 0 # yes, we exit the script here, dump_protocols_from_database will call all the other functions on its own
  fi

  if [[ "$INSTALL_APPLICATIONS" == "yes" ]] && [[ -n "$SESSION_ID" ]] && [[ -n "$MASTER_SESSION_ID" ]]; then
    # if both session IDs are set, this is a call from ControlService to install applications
    if [[ "$EXECUTE_DATABASE_OPS" == *"seed"* ]]; then
      # it contains "seed"
      # it means it is a call from the new ControlService:
      #   starting with version bodhi 1.2.x we must seed every time we do a software update
      #   because preset protocols evolve over time,
      #   and we want to keep the previous versions of the factory preset protocols
      #   so we don't seed only once but we might seed multiple times
      :
    else
      # it is a call from the old ControlService
      # and yet we must seed every time
      if [[ -z "$EXECUTE_DATABASE_OPS" ]] || [[ "$EXECUTE_DATABASE_OPS" == "nop" ]]; then
        EXECUTE_DATABASE_OPS="seed"
      else
        EXECUTE_DATABASE_OPS="$EXECUTE_DATABASE_OPS+seed"
      fi
    fi
  fi

  if [[ "$INSTALL_APPLICATIONS" == "yes" ]] || [[ "$INSTALL_PYRUNNER" == "yes" ]]; then
    echo ""
    echo "HOST_FQDN=${HOST_FQDN}"
    echo ""

    install_applications

    echo ""
    echo "see log at: $run_scripts_log_file"
    echo ""

    exit 0 # yes, we exit the script here, install_applications will call all the other functions on its own
  fi

  if [[ "$REINITIALIZE_INSTRUMENT" == "yes" ]]; then
    reinitialize_instrument
    
    echo ""
    echo "see log at: $run_scripts_log_file"
    echo ""

    exit 0 # yes, we exit the script here, reinitialize_instrument will call all the other functions on its own
  fi

  if [[ "$SET_LOCALIZATION" == "yes" ]] || \
    [[ "$UPDATE_HOST_PUBLIC_NAME" == "yes" ]] || \
    [[ "$EXECUTE_DATABASE_OPS" != "nop" ]] || \
    [[ "$UPDATE_PROTOCOLS" == "yes" ]] \
  ; then
    STOP_SERVICES="yes-all-except-control-pyrunner"
    START_SERVICES="yes-all-except-control-pyrunner"
  fi

  if [[ "$RESTART_SERVICES" == "yes" ]] || \
    [[ "$CREATE_SERVICES" == "yes" ]] || [[ "$CREATE_SERVICES" == "overwrite" ]] \
  ; then
    STOP_SERVICES="yes"
    START_SERVICES="yes"
  fi

  if [[ "$STOP_SERVICES" != "no" ]]; then
    stop_services $STOP_SERVICES
  fi

  if [[ "$STOP_APPS_PROCS" == "yes" ]]; then
    stop_apps_processes
  fi

  if [[ "$STOP_CONTROL_PROCS" == "yes" ]]; then
    stop_control_processes
  fi

  if [[ "$SET_LOCALIZATION" == "yes" ]]; then
    set_localization
  fi

  if [[ "$CREATE_SERVICES" == "yes" ]] || [[ "$CREATE_SERVICES" == "overwrite" ]]; then
    create_services
  fi

  if [[ "$UPDATE_HOST_PUBLIC_NAME" == "yes" ]]; then
    # in the current running deployment
    update_current_host_public_name
  fi

  if [[ "$EXECUTE_DATABASE_OPS" != "nop" ]]; then
    execute_database_ops
  fi
  
  if [[ "$UPDATE_PROTOCOLS" == "yes" ]]; then
    update_protocols_in_database "$UPDATE_PROTOCOLS_FILTER" "$UPDATE_PROTOCOLS_STATEMENT"
  fi

  if [[ "$START_CONTROL_PROCS" == "yes" ]]; then
    start_control_processes
  fi

  if [[ "$START_APPS_PROCS" == "yes" ]]; then
    start_app_processes
  fi

  if [[ "$START_SERVICES" != "no" ]]; then
    start_services $START_SERVICES
  fi
    
  echo ""
  echo "see log at: $run_scripts_log_file"
  echo ""
}

# Function to display usage information
usage() {
  echo "Usage: $0 [ options ]"
  echo "  --help                              Display this help message and exit"
  echo "  --version                           Display the script version and exit"
  echo "  --framework-language                Framework (default: $FRAMEWORK_LANGUAGE) : [ dotnet | nodejs ]"
  echo "  --install-applications              Install application (default: $INSTALL_APPLICATIONS) : [ no | yes ]"
  echo "  --install-applications-filename     Install application filename (default: $INSTALL_APPLICATIONS_FILENAME)"
  echo "  --install-pyrunner                  Install PyRunner (default: $INSTALL_PYRUNNER) : [ no | yes ]"
  echo "  --pyrunner-install-options          PyRunner install options (default: $PYRUNNER_INSTALL_OPTIONS)"
  echo "  --host-public-name                  Host public name (default: $HOST_PUBLIC_NAME)"
  echo "  --update-host-public-name           Update host public name (default: $UPDATE_HOST_PUBLIC_NAME) : [ no | yes ]"
  echo "  --create-services                   Create services (default: $CREATE_SERVICES): [ no | yes | overwrite ]"
  echo "  --stop-services                     Stop services (default: $STOP_SERVICES): [ no | yes | yes-all | yes-all-except-control | yes-all-except-control-pyrunner ]"
  echo "  --start-services                    Start services (default: $START_SERVICES): [ no | yes | yes-all | yes-all-except-control | yes-all-except-control-pyrunner ]"
  echo "  --restart-services                  Restart services (default: $RESTART_SERVICES): [ no | yes ]"
  echo "  --stop-apps-processes               Stop bodhi-apps service processes (default: $STOP_APPS_PROCS): [ no | yes ]"
  echo "  --start-apps-processes              Start bodhi-apps service processes (default: $START_APPS_PROCS): [ no | yes ]"
  echo "  --stop-control-processes            Stop bodhi-control service processes (default: $STOP_CONTROL_PROCS): [ no | yes ]"
  echo "  --start-control-processes           Start bodhi-control service processes (default: $START_CONTROL_PROCS): [ no | yes ]"
  echo "  --execute-database-ops              Database operations (default: $EXECUTE_DATABASE_OPS) : [ nop | delete | recreate | reinitialize | migrate | seed | migrate+seed ]"
  echo "  --locale-language                   Locale language (default: $LOCALE_LANG): [ en_US.UTF-8 | ... ]"
  echo "  --time-zone                         Timezone (default: $TIME_ZONE): [ UTC | ... ]"
  echo "  --set-localization                  Set localization (default: $SET_LOCALIZATION): [ no | yes ]"
  echo "  --generate-service-report           Generate Service Report (default: $GENERATE_SERVICE_REPORT): [ no | yes ]"
  echo "  --dump-protocols                    Dump Protocols from database (default: $DUMP_PROTOCOLS): [ no | yes ]"
  echo "  --dump-protocols-filter             SQL WHERE clause to filter protocols (default: '$DUMP_PROTOCOLS_FILTER')"
  echo "  --update-protocols                  Update Protocols in database (default: $UPDATE_PROTOCOLS): [ no | yes ]"
  echo "  --update-protocols-filter           SQL WHERE clause to filter protocols for update (default: '$UPDATE_PROTOCOLS_FILTER')"
  echo "  --update-protocols-statement        SQL SET clause for protocol update (default: '$UPDATE_PROTOCOLS_STATEMENT')"
  echo "  --delete-all                        Remove all bodhi files and services from the system (default: $DELETE_ALL): [ no | yes ]"
  echo "  --reinitialize-instrument           System will rehome hardware, restart pyrunner, PostgreSQL, and other instrument software. (default: $REINITIALIZE_INSTRUMENT): [ no | yes ]"
  echo "  --session-id                        Session ID (default: $SESSION_ID)"
  echo "  --master-session-id                 Master Session ID (default: $MASTER_SESSION_ID)"
  echo "  --client-datetime                   Client DateTime (default: $CLIENT_DATETIME)"
}


. ./lib/consts.sh
. ./lib/helpers.sh
. ./lib/system.sh
. ./lib/services.sh
. ./lib/postgres.sh
. ./lib/applications-settings.sh
. ./lib/applications-report.sh
. ./lib/pyrunner.sh
. ./lib/install-applications.sh
. ./lib/service-report.sh
. ./lib/delete-all.sh

# call the main function
main "$@"