# Function to change the last octet of an IP address by an offset
chg_ip_addr() {
    local ip_addr="$1"
    local offset="$2"

    # Validate input
    if [[ -z "$ip_addr" || -z "$offset" ]]; then
        echo "ERROR: Usage: chg_ip_addr <ip_addr> <offset>" >&2
        return 1
    fi

    # Tokenize IP address
    IFS='.' read -r -a ip_parts <<< "$ip_addr"
    local nLen=${#ip_parts[@]}
    if [[ $nLen -ne 4 ]]; then
        echo "ERROR: Invalid IP address format: $ip_addr" >&2
        return 1
    fi

    # Convert to integers and apply offset to last octet
    local result_parts=()
    for ((n=0; n<nLen; n++)); do
        local part_val
        if [[ $n -eq $((nLen-1)) ]]; then
            part_val=$(( ${ip_parts[n]:-0} + offset ))
            if (( part_val > 255 )); then
                echo "ERROR: IP-Address too large: $ip_addr Offset = $offset > 255" >&2
                return 1
            fi
        else
            part_val=$(( ${ip_parts[n]:-0} ))
        fi
        result_parts+=("$part_val")
    done

    local new_ip="${result_parts[0]}.${result_parts[1]}.${result_parts[2]}.${result_parts[3]}"
    echo "$new_ip"
    # Optionally log
    # echo "ChgIPAddr $ip_addr to $new_ip" >&2
    return 0
}

# Function to get the non-loopback IP address of the current machine
# Returns the first non-loopback, non-docker IP address found
# Returns empty string if no suitable IP address is found
get_ip_address() {
    # Try to get IP address using different methods, in order of preference
    
    # Method 1: Use hostname -I (most reliable on most Linux systems)
    if command -v hostname &> /dev/null; then
        local ip_list=$(hostname -I 2>/dev/null)
        if [[ -n "$ip_list" ]]; then
            # hostname -I returns all IPs, pick the first non-loopback one
            for ip in $ip_list; do
                # Skip loopback addresses (127.x.x.x) and Docker bridge IPs (172.17.x.x)
                if [[ ! "$ip" =~ ^127\. ]] && [[ ! "$ip" =~ ^172\.17\. ]]; then
                    echo "$ip"
                    return 0
                fi
            done
        fi
    fi
    
    # Method 2: Use ip command if available
    if command -v ip &> /dev/null; then
        local ip_addr=$(ip route get 8.8.8.8 2>/dev/null | grep -oP 'src \K[^ ]+' | head -1)
        if [[ -n "$ip_addr" ]] && [[ ! "$ip_addr" =~ ^127\. ]]; then
            echo "$ip_addr"
            return 0
        fi
    fi
    
    # Method 3: Parse ifconfig output
    if command -v ifconfig &> /dev/null; then
        local ip_addr=$(ifconfig 2>/dev/null | grep -E 'inet [0-9]' | grep -v '127.0.0.1' | grep -v '172.17.' | head -1 | awk '{print $2}' | sed 's/addr://')
        if [[ -n "$ip_addr" ]]; then
            echo "$ip_addr"
            return 0
        fi
    fi
    
    # Method 4: Use /proc/net/route as fallback
    if [[ -f /proc/net/route ]]; then
        local interface=$(awk '/^[^I]/ { if ($2 == "00000000") print $1 }' /proc/net/route | head -1)
        if [[ -n "$interface" ]]; then
            local ip_addr=$(ip addr show "$interface" 2>/dev/null | grep -oP 'inet \K[^/]+' | head -1)
            if [[ -n "$ip_addr" ]] && [[ ! "$ip_addr" =~ ^127\. ]]; then
                echo "$ip_addr"
                return 0
            fi
        fi
    fi
    
    # If all methods fail, return empty string
    echo ""
    return 0
}

# Function to convert a string to an integer with default value
# Parameters:
#   str - the string to convert
#   default - the default value to return if conversion fails (optional, defaults to 0)
# Returns the integer value or the default if conversion fails
str_to_int() {
    local str="$1"
    local default="${2:-0}"
    
    # Check if string is empty
    if [[ -z "$str" ]]; then
        echo "$default"
        return 0
    fi
    
    # Check if string is a valid integer (optional minus sign followed by digits only)
    if [[ "$str" =~ ^-?[0-9]+$ ]]; then
        echo "$str"
    else
        echo "$default"
    fi
}

# Function to check if a string is a number (integer or float)
is_str_a_number() {
    local str="$1"
    if [[ "$str" =~ ^-?[0-9]+([.][0-9]+)?$ ]]; then
        echo "true"
    else
        echo "false"
    fi
}
# Function to multiply two string arguments if they are numbers
str_multiply() {
    local str1="$1"
    local str2="$2"

    # Check for missing parameters
    if [[ -z "$str1" || -z "$str2" ]]; then
        echo "ERROR: Two parameters required" >&2
        return 1
    fi

    # Check if both parameters are numbers (integer or float)
    if ! [[ "$str1" =~ ^-?[0-9]+([.][0-9]+)?$ ]] || ! [[ "$str2" =~ ^-?[0-9]+([.][0-9]+)?$ ]]; then
        echo "ERROR: Both parameters must be numbers" >&2
        return 1
    fi

    # Multiply using bc for float support
    local result=$(echo "$str1 * $str2" | bc)
    echo "$result"
}

# Function to read a specific section from .iss or .ini files
# Returns all variables as key-value pairs
read_ini_section() {
    local section_name="$1"
    local file_path="$2"
    
    # Check if both parameters are provided
    if [[ -z "$section_name" ]]; then
        echo "ERROR: Section name is required" >&2
        echo "Usage: read_ini_section <section_name> <file_path>" >&2
        return 1
    fi
    
    if [[ -z "$file_path" ]]; then
        echo "ERROR: File path parameter is required" >&2
        echo "Usage: read_ini_section <section_name> <file_path>" >&2
        return 1
    fi
    
    # Check if the file exists
    if [[ ! -f "$file_path" ]]; then
        echo "ERROR: File not found at: $file_path" >&2
        return 1
    fi

    # Read the specified section and extract key=value pairs
    # Using sed and grep for a more reliable approach
    
    # Create a temporary file to store results
    local temp_result=$(mktemp)
    
    # Extract section content - from [section_name] to next [section] or end of file
    sed -n "/^\[${section_name}\]/,/^\[.*\]/{
        /^\[${section_name}\]/d
        /^\[.*\]/d
        p
    }" "$file_path" > "$temp_result"
    
    # If no content found, try extracting to end of file (section at end)
    if [[ ! -s "$temp_result" ]]; then
        sed -n "/^\[${section_name}\]/,\$p" "$file_path" | tail -n +2 > "$temp_result"
    fi
    
    # echo "DEBUG: Section content for '$section_name':" >&2
    # cat "$temp_result" >&2
    
    # Process each line to extract key=value pairs
    local output_file=$(mktemp)
    while IFS= read -r line || [[ -n "$line" ]]; do
        # Skip empty lines and comments
        if [[ "$line" =~ ^[[:space:]]*$ ]] || [[ "$line" =~ ^[[:space:]]*\; ]] || [[ "$line" =~ ^[[:space:]]*# ]]; then
            continue
        fi
        
        # Only process lines with = sign
        if [[ "$line" == *"="* ]]; then
            # Extract key and value
            key=$(echo "$line" | cut -d'=' -f1 | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
            value=$(echo "$line" | cut -d'=' -f2- | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
            
            # Remove surrounding quotes if they exist
            value=$(echo "$value" | sed 's/^"//;s/"$//')
            
            echo "${key}=${value}" >> "$output_file"
        fi
    done < "$temp_result"
    
    # Output the results
    cat "$output_file"
    
    # Clean up temporary files
    rm -f "$temp_result" "$output_file"
}

# Function to get a specific variable from a specific section
get_ini_value() {
    local var_name="$1"
    local section_name="$2"
    local file_path="$3"
    local default_value="${4:-}"

    # Check if all required parameters are provided
    if [[ -z "$var_name" ]]; then
        echo "ERROR: Variable name is required" >&2
        echo "Usage: get_ini_value <variable_name> <section_name> <file_path> [default_value]" >&2
        return 1
    fi

    if [[ -z "$section_name" ]]; then
        echo "ERROR: Section name is required" >&2
        echo "Usage: get_ini_value <variable_name> <section_name> <file_path> [default_value]" >&2
        return 1
    fi

    if [[ -z "$file_path" ]]; then
        echo "ERROR: File path is required" >&2
        echo "Usage: get_ini_value <variable_name> <section_name> <file_path> [default_value]" >&2
        return 1
    fi

    # Get all variables from the section
    local section_vars=$(read_ini_section "$section_name" "$file_path")

    # Find the line for the variable (if present) - support spaces around =
    local line=$(echo "$section_vars" | grep -m1 "^${var_name}[[:space:]]*=")

    if [[ -n "$line" ]]; then
        # Variable exists (may be empty) - extract value after =
        local value="${line#*=}"
        # Trim leading and trailing whitespace from value
        value=$(echo "$value" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        echo "$value"
    else
        # Variable not found, return default_value
        echo "$default_value"
    fi
}

# Function to set a variable in a specific section of .iss or .ini files
# Creates the section if it doesn't exist
set_ini_value() {
    local var_name="$1"
    local var_value="$2"
    local section_name="$3"
    local file_path="$4"
    
    # Check if all parameters are provided
    if [[ -z "$var_name" ]]; then
        echo "ERROR: Variable name is required" >&2
        echo "Usage: set_ini_value <variable_name> <variable_value> <section_name> <file_path>" >&2
        return 1
    fi
    
    # Note: var_value can be empty, so we check if it's unset instead of empty
    if [[ $# -lt 2 ]]; then
        echo "ERROR: Variable value parameter is required (can be empty)" >&2
        echo "Usage: set_ini_value <variable_name> <variable_value> <section_name> <file_path>" >&2
        return 1
    fi
    
    if [[ -z "$section_name" ]]; then
        echo "ERROR: Section name is required" >&2
        echo "Usage: set_ini_value <variable_name> <variable_value> <section_name> <file_path>" >&2
        return 1
    fi
    
    if [[ -z "$file_path" ]]; then
        echo "ERROR: File path is required" >&2
        echo "Usage: set_ini_value <variable_name> <variable_value> <section_name> <file_path>" >&2
        return 1
    fi
    
    # Create file if it doesn't exist
    if [[ ! -f "$file_path" ]]; then
        touch "$file_path"
    fi
    
    # Check if section exists
    if ! grep -q "^\[$section_name\]" "$file_path"; then
        # Add section at the end of file
        echo "" >> "$file_path"
        echo "[$section_name]" >> "$file_path"
    fi
    
    # Check if the variable already exists in the section
    local temp_file=$(mktemp)
    local in_section=0
    local var_found=0
    
    while IFS= read -r line || [[ -n "$line" ]]; do
        # Check if we're entering target section
        if [[ "$line" =~ ^[[:space:]]*\[$section_name\][[:space:]]*$ ]]; then
            in_section=1
            echo "$line" >> "$temp_file"
            continue
        fi
        
        # Check if we're entering another section
        if [[ "$line" =~ ^[[:space:]]*\[.*\][[:space:]]*$ ]] && [[ $in_section -eq 1 ]]; then
            # If we're leaving target section and variable wasn't found, add it
            if [[ $var_found -eq 0 ]]; then
                echo "$var_name=$var_value" >> "$temp_file"
                var_found=1
            fi
            in_section=0
        fi
        
        # If we're in target section, check for existing variable
        if [[ $in_section -eq 1 ]]; then
            # Check if this line contains our variable (handle whitespace around = sign)
            if [[ "$line" =~ ^[[:space:]]*$var_name[[:space:]]*=[[:space:]]* ]]; then
                # Replace existing variable
                echo "$var_name=$var_value" >> "$temp_file"
                var_found=1
                continue
            fi
        fi
        
        # Copy line as-is
        echo "$line" >> "$temp_file"
        
    done < "$file_path"
    
    # If we reached end of file while in target section and variable wasn't found, add it
    if [[ $in_section -eq 1 ]] && [[ $var_found -eq 0 ]]; then
        echo "$var_name=$var_value" >> "$temp_file"
    fi
    
    # Replace original file with modified content
    mv "$temp_file" "$file_path"
}

# Function to remove a variable from a specific section of .iss or .ini files
# Parameters:
#   var_name - the name of the variable to remove
#   section_name - the name of the section
#   file_path - path to the .ini/.iss file
# Returns:
#   0 - success (variable removed or didn't exist)
#   1 - error (invalid parameters or file issues)
remove_ini_value() {
    local var_name="$1"
    local section_name="$2"
    local file_path="$3"
    
    # Check if all parameters are provided
    if [[ -z "$var_name" ]]; then
        echo "ERROR: Variable name is required" >&2
        echo "Usage: remove_ini_value <variable_name> <section_name> <file_path>" >&2
        return 1
    fi
    
    if [[ -z "$section_name" ]]; then
        echo "ERROR: Section name is required" >&2
        echo "Usage: remove_ini_value <variable_name> <section_name> <file_path>" >&2
        return 1
    fi
    
    if [[ -z "$file_path" ]]; then
        echo "ERROR: File path is required" >&2
        echo "Usage: remove_ini_value <variable_name> <section_name> <file_path>" >&2
        return 1
    fi
    
    # Check if file exists
    if [[ ! -f "$file_path" ]]; then
        echo "ERROR: File not found at: $file_path" >&2
        return 1
    fi
    
    # Check if section exists
    if ! grep -q "^\[$section_name\]" "$file_path"; then
        # Section doesn't exist, nothing to remove
        return 0
    fi
    
    # Create temporary file for processing
    local temp_file=$(mktemp)
    local in_section=0
    
    while IFS= read -r line || [[ -n "$line" ]]; do
        # Check if we're entering target section
        if [[ "$line" =~ ^[[:space:]]*\[$section_name\][[:space:]]*$ ]]; then
            in_section=1
            echo "$line" >> "$temp_file"
            continue
        fi
        
        # Check if we're entering another section
        if [[ "$line" =~ ^[[:space:]]*\[.*\][[:space:]]*$ ]] && [[ $in_section -eq 1 ]]; then
            in_section=0
        fi
        
        # If we're in target section, check if this is the variable to remove
        if [[ $in_section -eq 1 ]]; then
            # Check if this line contains our variable (handle whitespace around = sign)
            if [[ "$line" =~ ^[[:space:]]*$var_name[[:space:]]*=[[:space:]]* ]]; then
                # Skip this line (don't write it to temp file)
                continue
            fi
        fi
        
        # Copy line as-is
        echo "$line" >> "$temp_file"
        
    done < "$file_path"
    
    # Replace original file with modified content
    mv "$temp_file" "$file_path"
    
    return 0
}

# Function to extract a specific option from a space-delimited options string
# Parameters:
#   option_name - the name of the option to look for (leading slash/backslash is automatically normalized)
#   options - space-delimited string containing flags and key=value pairs
# Returns:
#   - For flags (like RPISIMULATION, FLASH, /TEST, \TEST): returns "true" if present, empty string if not
#   - For key=value pairs (like INVENTORY=a,b,c, /CONFIG=value, \CONFIG=value): returns the value part if present, empty string if not
#   - Leading slashes or backslashes on both option_name and options are ignored for matching
#   - Examples: option_name="/TEST" matches options "TEST", "/TEST", "\TEST"
#   - Always returns exit code 0
get_option_from_options() {
    local option_name="$1"
    local options="$2"
    
    # Check if parameters are provided
    if [[ -z "$option_name" ]]; then
        echo "ERROR: Option name is required" >&2
        echo "Usage: get_option_from_options <option_name> <options>" >&2
        return 1
    fi
    
    # Validate that option_name is not just a slash or backslash
    if [[ "$option_name" == "/" ]] || [[ "$option_name" == "\\" ]]; then
        echo "ERROR: Option name cannot be just a slash or backslash" >&2
        echo "Usage: get_option_from_options <option_name> <options>" >&2
        return 1
    fi
    
    if [[ -z "$options" ]]; then
        echo ""
        return 0
    fi
    
    # Normalize option_name by removing leading slash or backslash if present
    local clean_option_name="$option_name"
    if [[ "$option_name" == /* ]] || [[ "$option_name" == \\* ]]; then
        clean_option_name="${option_name#?}"
    fi
    
    # Validate that clean_option_name is not empty after normalization
    if [[ -z "$clean_option_name" ]]; then
        echo "ERROR: Option name cannot be empty after removing leading slash/backslash" >&2
        echo "Usage: get_option_from_options <option_name> <options>" >&2
        return 1
    fi
    
    # Convert options string to array for easier processing
    local options_array=($options)
    
    # Look for the option in the array
    for option in "${options_array[@]}"; do
        # Check if it's a key=value pair
        if [[ "$option" == *"="* ]]; then
            # Extract key and value
            local key=$(echo "$option" | cut -d'=' -f1)
            local value=$(echo "$option" | cut -d'=' -f2-)
            
            # Remove leading slash or backslash from key if present for comparison
            local clean_key="$key"
            if [[ "$key" == /* ]] || [[ "$key" == \\* ]]; then
                clean_key="${key#?}"
            fi
            
            # If key matches our normalized option_name, return the value
            if [[ "$clean_key" = "$clean_option_name" ]]; then
                echo "$value"
                return 0
            fi
        else
            # It's a flag - remove leading slash or backslash if present for comparison
            local clean_option="$option"
            if [[ "$option" == /* ]] || [[ "$option" == \\* ]]; then
                clean_option="${option#?}"
            fi
            
            # Check if it matches our normalized option_name
            if [[ "$clean_option" = "$clean_option_name" ]]; then
                echo "true"
                return 0
            fi
        fi
    done
    
    # Option not found - return empty string with exit code 0
    echo ""
    return 0
}

# Function to get current server timestamp with milliseconds
# Returns:
#   Current server timestamp in format: seconds.milliseconds (e.g., "1693915845.123")
get_server_timestamp() {
    local server_timestamp
    
    if date --version 2>/dev/null | grep -q GNU; then
        # GNU date (Linux)
        server_timestamp=$(date +%s.%3N)
    else
        # BSD date (macOS) or other systems - approximate milliseconds
        local base_ts=$(date +%s)
        local nano_part=$(date +%N 2>/dev/null)
        if [[ -n "$nano_part" ]] && [[ "$nano_part" != "N" ]]; then
                # Convert nanoseconds to milliseconds (first 3 digits, padded)
                local milli_raw="${nano_part:0:3}"
                local milli_part=$(printf "%03d" $((10#$milli_raw)))
                server_timestamp="${base_ts}.${milli_part}"
        else
            # Fallback: use current time in milliseconds since epoch (approximation)
            server_timestamp="${base_ts}.$(( $(date +%s) % 1000 ))"
        fi
    fi
    
    echo "$server_timestamp"
}

# Function to check for datetime skew between client and server
# Parameters:
#   $1 - client_datetime_str: ISO UTC datetime string from PowerShell (e.g., "2025-09-05T14:30:45.123Z")
#   $2 - server_timestamp: (optional) Server timestamp in seconds.milliseconds format. If not provided, current time is used.
# Outputs:
#   Displays the time difference in seconds and human-readable format
# Returns:
#   0 on success, 1 on error
check_for_datetime_skew() {
    local client_datetime_str="$1"
    local server_timestamp="$2"
    
    # Validate input
    if [[ -z "$client_datetime_str" ]]; then
        echo "WARNING: No client datetime provided for skew check"
        return 1
    fi
    if [[ -z "$server_timestamp" ]]; then
        echo "WARNING: No server timestamp provided, using current time"
        return 1
    fi
    
    # Parse the ISO datetime string (format: 2025-09-05T14:30:45.123Z)
    # Remove the 'Z' and split into date and time parts
    local datetime_clean="${client_datetime_str%Z}"
    local date_part="${datetime_clean%T*}"
    local time_part="${datetime_clean#*T}"
    
    # Split date into components (YYYY-MM-DD)
    local year="${date_part%-*-*}"
    local month_day="${date_part#*-}"
    local month="${month_day%-*}"
    local day="${month_day#*-}"
    
    # Split time into components (HH:MM:SS.sss)
    local hour_min_sec="${time_part%.*}"
    local milliseconds="${time_part#*.}"
    local hour="${hour_min_sec%:*:*}"
    local min_sec="${hour_min_sec#*:}"
    local minute="${min_sec%:*}"
    local second="${min_sec#*:}"
    
    # Convert client datetime to Unix timestamp with milliseconds
    # First get the base timestamp without milliseconds
    local client_timestamp_base
    if command -v date &> /dev/null; then
        # Try GNU date format first
        client_timestamp_base=$(date -u -d "${year}-${month}-${day} ${hour}:${minute}:${second}" +%s 2>/dev/null)
        
        # If GNU date failed, try BSD date format (macOS)
        if [[ $? -ne 0 ]] || [[ -z "$client_timestamp_base" ]]; then
            client_timestamp_base=$(date -u -j -f "%Y-%m-%d %H:%M:%S" "${year}-${month}-${day} ${hour}:${minute}:${second}" +%s 2>/dev/null)
        fi
    else
        echo "ERROR: 'date' command not available for datetime parsing"
        return 1
    fi
    
    if [[ -z "$client_timestamp_base" ]]; then
        echo "ERROR: Failed to parse client datetime: $client_datetime_str"
        return 1
    fi
    
    # Add milliseconds to get fractional seconds
    local client_timestamp="${client_timestamp_base}.${milliseconds}"
    
    # Calculate the difference in seconds (floating point)
    local diff_seconds
    diff_seconds=$(echo "$server_timestamp - $client_timestamp" | bc -l 2>/dev/null)
    
    if [[ -z "$diff_seconds" ]]; then
        # Fallback if bc is not available - use millisecond-precision arithmetic
        local client_int=${client_timestamp%.*}
        local client_milli=${client_timestamp#*.}
        local server_int=${server_timestamp%.*}
        local server_milli=${server_timestamp#*.}
        
        # Ensure millisecond parts are 3 digits (pad with zeros if needed)
        client_milli=$(printf "%03d" $((10#$client_milli)))
        server_milli=$(printf "%03d" $((10#$server_milli)))

        # Convert to total milliseconds for calculation
        local client_total_ms=$((client_int * 1000 + 10#$client_milli))
        local server_total_ms=$((server_int * 1000 + 10#$server_milli))
        
        # Calculate difference in milliseconds
        local diff_ms=$((server_total_ms - client_total_ms))
        
        # Convert back to seconds with 3 decimal places
        local diff_int=$((diff_ms / 1000))
        local diff_remainder=$((diff_ms % 1000))
        
        # Handle negative numbers properly
        if [[ $diff_ms -lt 0 ]]; then
            diff_int=$((-diff_ms / 1000))
            diff_remainder=$((-diff_ms % 1000))
            if [[ $diff_remainder -ne 0 ]]; then
                diff_int=$((diff_int + 1))
                diff_remainder=$((1000 - diff_remainder))
            fi
            diff_seconds="-${diff_int}.$(printf "%03d" $diff_remainder)"
        else
            diff_seconds="${diff_int}.$(printf "%03d" $diff_remainder)"
        fi
    fi
    
    # Display the raw difference
    echo "Time skew analysis:"
    echo "  Client datetime:    $client_datetime_str"
    echo "  Server datetime:    $(date -u +%Y-%m-%dT%H:%M:%S.%3NZ 2>/dev/null || date -u +%Y-%m-%dT%H:%M:%SZ)"
    echo "  Difference:         ${diff_seconds} seconds"
    
    # Convert to human-readable format
    local abs_diff_seconds
    abs_diff_seconds=$(echo "$diff_seconds" | sed 's/^-//')  # Remove negative sign for calculations
    
    # Check if we have a negative difference (client is ahead)
    local sign=""
    if [[ "$diff_seconds" == -* ]]; then
        sign="Client is ahead by: "
    else
        sign="Server is ahead by: "
    fi
    
    # Convert to integer seconds for calculation
    local total_seconds=${abs_diff_seconds%.*}
    local milliseconds_part=${abs_diff_seconds#*.}
    
    # Handle case where there's no decimal part
    if [[ "$milliseconds_part" == "$abs_diff_seconds" ]]; then
        milliseconds_part="000"
    fi
    
    # Calculate time components
    local days=$((total_seconds / 86400))
    local remaining=$((total_seconds % 86400))
    local hours=$((remaining / 3600))
    remaining=$((remaining % 3600))
    local minutes=$((remaining / 60))
    local seconds=$((remaining % 60))
    
    # Build human-readable string
    local human_readable=""
    if [[ $days -gt 0 ]]; then
        human_readable="${days} day(s), "
    fi
    if [[ $hours -gt 0 ]]; then
        human_readable="${human_readable}${hours} hour(s), "
    fi
    if [[ $minutes -gt 0 ]]; then
        human_readable="${human_readable}${minutes} minute(s), "
    fi
    
    # Always show seconds and milliseconds
    human_readable="${human_readable}${seconds}.${milliseconds_part:0:3} second(s)"
    
    echo "  Human readable:     ${sign}${human_readable}"
    
    # Provide some context about the skew
    local abs_total_seconds=$total_seconds
    if [[ $abs_total_seconds -gt 300 ]]; then  # More than 5 minutes
        echo "  WARNING: Significant time skew detected (>5 minutes)"
        echo "           This will negatively affect the Authentication/Authorization [time sensitive]"
    elif [[ $abs_total_seconds -gt 60 ]]; then  # More than 1 minute
        echo "  NOTICE: Notable time skew detected (>1 minute)"
    elif [[ $abs_total_seconds -gt 5 ]]; then   # More than 5 seconds
        echo "  INFO: Minor time skew detected (>5 seconds)"
    else
        echo "  INFO: Time skew is within acceptable range"
    fi
    echo "Note: this is an aproximation due to the time it takes to run the scripts [~0.5-1.5 seconds]"
    
    return 0
}

# Function to check if an item is in inventory
is_in_inventory() {
    local inventory="$1"
    local item="$2"

    # Validate parameter - return "false" on error to avoid breaking conditional logic
    if [[ -z "$item" ]]; then
        echo "   ERROR: is_in_inventory() requires an item parameter!" >&2
        echo "false"
        return 1
    fi

    local inventory_lower=$(echo "$inventory" | tr '[:upper:]' '[:lower:]')
    local item_lower=$(echo "$item" | tr '[:upper:]' '[:lower:]')
    
    # Check if the lowercase item exists in the lowercase inventory string
    if [[ "$inventory_lower" == *"$item_lower"* ]]; then
        echo "true"
    else
        echo "false"
    fi
    
    return 0
}
