#!/bin/bash

# ==============================================================================
# SCRIPT DESCRIPTION:
# This script merges the contents of the '[FI_Bottom]' section into the
# '[GC_Params]' section within specified configuration files.
#
# Key features:
# 1. Handles multiple configuration files defined in the 'FILES' array.
# 2. If a key exists in both sections, the value from '[FI_Bottom]' overrides
#    the value in '[GC_Params]'.
# 3. The original '[FI_Bottom]' header and its data are completely removed.
# 4. Other headers (e.g., '[Positions]') and their data are preserved in their
#    original location and order.
# 5. Uses standard POSIX 'awk' commands for maximum compatibility across systems.
# ==============================================================================


if [ $# -eq 0 ]; then
    echo "This script takes argument username for PyRunner config file paths."
    exit 1
fi

pyrunner_user=$1

# --- List of Files ---
FILES=(
  "/home/$pyrunner_user/.config/pyrunner/ScanTable.adj"
  "/home/$pyrunner_user/.config/pyrunner/MeasurementUnit.adj"
)
# -------------------------------------------------

# Loop through all files in the hardcoded FILES array
for FILE_PATH in "${FILES[@]}"; do

    if [[ ! -f "$FILE_PATH" ]]; then
        echo "Error: File not found at $FILE_PATH - skipping."
        continue
    fi

    echo "Processing file: $FILE_PATH"

    # Use a temporary file for atomic writing
    FINAL_TEMP="${FILE_PATH}.tmp"
    trap 'rm -f "$FINAL_TEMP"' EXIT

    # Use a single, powerful AWK command to handle everything using POSIX standard functions:

    awk '
    BEGIN {
        FS="="
        OFS="="
        # Array to store all unique keys and values
        keys_array["_placeholder_"]=""
        # Flag to track if [GC_Params] header was ever seen
        gc_header_found = 0
        in_gc = 0
        in_fb = 0
    }

    # Helper function to strip whitespace
    function strip(str) {
        sub(/^[[:space:]]*/, "", str)
        sub(/[[:space:]]*$/, "", str)
        return str
    }

    /\[GC_Params\]/ {
        in_gc = 1
        in_fb = 0
        gc_header_found = 1
        print $0
        next
    }

    /\[FI_Bottom\]/ {
        in_fb = 1
        in_gc = 0
        next # Skip printing the [FI_Bottom] header
    }

    /\[.*\]/ { # Any other header ends current section flags
        in_gc = 0
        in_fb = 0
        print $0
        next
    }

    # Process key-value pairs and blank lines
    {
        if ($0 ~ /^\[/) {
           # Handled by the /\[.*\]/ blocks above
        } else if (!/^[[:space:]]*$/) {
            # Process as a key-value line
            key = strip($1)
            value = strip($2)

            if (in_gc || in_fb) {
                if (key != "") {
                    keys_array[key] = value # Store/Overwrite key
                }
            } else {
                # Print non-target section lines immediately
                print $0
            }
        } else {
            # Print blank lines immediately if they are not within target sections
            if (!in_gc && !in_fb) {
                print $0
            }
        }
    }


    END {
        # If [GC_Params] was found, we already printed the header and will rely on the
        # main loop having printed the rest of the file *except* the key-value data.
        # We only need to append the combined key list *if* the header was found during the run.
        if (gc_header_found) {
            # Print all unique key-value pairs below the header location
            for (key in keys_array) {
                if (key != "_placeholder_") {
                    print key OFS keys_array[key]
                }
            }
        } else {
            # If [GC_Params] was NEVER found in the file, append the new section at the very end
            print "" # Add a newline for formatting
            print "[GC_Params]"
            for (key in keys_array) {
                 if (key != "_placeholder_") {
                    print key OFS keys_array[key]
                }
            }
        }
    }
    ' "$FILE_PATH" > "$FINAL_TEMP"

    # Overwrite the original file
    mv "$FINAL_TEMP" "$FILE_PATH"
    echo "[FI_Bottom] merged into [GC_Params] successfully for $FILE_PATH ."

done
