
allow_current_user_to_run_as_postgres_without_a_password() {
  echo "  check that user ${USER_NAME} is allowed to run as postgres without a password..."
  local sudoers_entry=$(grep -P "^${USER_NAME}\s+ALL=\(postgres:postgres\) NOPASSWD: ALL" /etc/sudoers)
  if [[ -z "$sudoers_entry" ]]
  then
      echo "  the required setting was not found in the sudoers file, adding it now..."
      echo "${USER_NAME}    ALL=(postgres:postgres) NOPASSWD: ALL" | tee -a /etc/sudoers  >/dev/null
  else
    echo "  user ${USER_NAME} is allowed to run as postgres without a password"
  fi
}

set_postgres_user_password() {
  echo "  set postgres user password..."  
  local set_postgres_user_password=$(sudo -u postgres psql -U postgres -tc "ALTER USER postgres with encrypted password 'postgres';" 2>&1)
  if [[ $? -eq 0 ]]; then
      echo "  set postgres user password completed successfully"
  else
      echo "  ERROR: failed to set the postgres user password, ${set_postgres_user_password} !!"
  fi
}

delete_database() {
  local db_name=$1
  local db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '$db_name'")
  if [[ ! -z "$db_exists" ]]; then
      echo "  database $db_name exists, drop connections, then delete the database..."
      sudo -u postgres psql -tc "\
        BEGIN; \
          SELECT pg_terminate_backend(pg_stat_activity.pid) \
          FROM pg_stat_activity \
          WHERE datname = '$db_name' AND pid <> pg_backend_pid(); \
        COMMIT; \
      " > /dev/null 2>&1
      sleep 10
      sudo -u postgres psql -tc "DROP DATABASE IF EXISTS \"$db_name\";" > /dev/null 2>&1
  fi
}

new_db=false
create_database() {
  local db_name=$1
  local db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '$db_name'")
  new_db=false
  if [[ ! -z "$db_exists" ]]; then
      echo "  database $db_name exists"
  else
      echo "  database $db_name does not exist, try to create it..."
      create_db_output=$(sudo -u postgres psql -U postgres -tc "CREATE DATABASE \"$db_name\"  WITH TEMPLATE template0 LC_COLLATE=\"${LOCALE_LANG}\" LC_CTYPE=\"${LOCALE_LANG}\";" 2>&1)
      db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '$db_name'")
      if [[ ! -z "$db_exists" ]]; then
          echo "  database $db_name was created successfully"
          new_db=true
      else
          #restart the postgresql service... and try again"
          restart_postgres_service
          sleep 15

          create_db_output=$(sudo -u postgres psql -U postgres -tc "CREATE DATABASE \"$db_name\"  WITH TEMPLATE template0 LC_COLLATE=\"${LOCALE_LANG}\" LC_CTYPE=\"${LOCALE_LANG}\";" 2>&1)
          db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '$db_name'")
          if [[ ! -z "$db_exists" ]]; then
              echo "  database $db_name was created successfully"
              new_db=true
          else
            echo "  ERROR: failed to create $db_name, ${create_db_output} !!"
            return 1
          fi
      fi
  fi
}

seed_database_in_nodejs() {
  local db_name=$1
  local new_db=$2

  if ! command -v curl >/dev/null 2>&1; then
    echo "  ERROR: curl is not available!"
    return 1
  fi

  env_file_path='/bodhi/run/current/apps/bodhi-api/.env'

  stop_services 

  echo "  disable authentication"
  local from=NX_PUBLIC_HOST=0.0.0.0
  local to=NX_PUBLIC_HOST=127.0.0.1
  sed -i "s|${from}|${to}|g" "${env_file_path}"
  from=NX_PUBLIC_REQUIRE_AUTH=true
  to=NX_PUBLIC_REQUIRE_AUTH=false
  sed -i "s|${from}|${to}|g" "${env_file_path}"

  start_bodhi_apps_service

  # wait just a bit for the API startup to finish
  sleep 30

  echo "  seed the database $db_name by calling the Api seed endpoints"
  local base_url="http://127.0.0.1/web"
  local endpoints=(
    "/mirrors/seed"
    "/optical-filters/seed"
    "/plate-types/seed"
    "/protocols/seed"
  )

  for endpoint in "${endpoints[@]}"; do
    local url="${base_url/}${endpoint}"
    echo "   call seed endpoint at ${endpoint}"
    local temp_file=$(mktemp)
    local retry_count=0
    local max_retries=3
    
    while [ $retry_count -lt $max_retries ]; do
      http_status=$(curl --request POST --url ${url} --silent --write-out "%{http_code}\n" --output $temp_file)
      response_body=$(cat $temp_file)
      
      # Check if status code is in 200s range
      if [[ "$http_status" =~ ^2[0-9][0-9]$ ]]; then
        echo "   response status: $http_status, body: $response_body"
        break
      else
        retry_count=$((retry_count + 1))
        if [ $retry_count -lt $max_retries ]; then
          echo "   response status: $http_status (attempt $retry_count/$max_retries), retrying in 5 seconds..."
          sleep 5
        else
          echo "   response status: $http_status, body: $response_body (failed after $max_retries attempts)"
        fi
      fi
    done
    
    rm -f $temp_file
  done

  stop_bodhi_apps_service

  echo "  re-enable authentication"
  local from=NX_PUBLIC_HOST=127.0.0.1
  local to=NX_PUBLIC_HOST=0.0.0.0
  sed -i "s|${from}|${to}|g" "${env_file_path}"
  from=NX_PUBLIC_REQUIRE_AUTH=false
  to=NX_PUBLIC_REQUIRE_AUTH=true
  sed -i "s|${from}|${to}|g" "${env_file_path}"

}

migration_table_exists=false
check_if_migration_table_exists() {
  local db_name=$1
  local table_exists=$(sudo -u postgres psql -d "${db_name}" -tc "SELECT EXISTS ( SELECT 1 FROM information_schema.tables WHERE table_schema = 'public' AND table_name = '__EFMigrationsHistory');")
  table_exists=$(echo $table_exists | xargs)
  if [[ "$table_exists" = "t" ]]; then
    migration_table_exists=true
    #echo "  __EFMigrationsHistory table exists"
  else
    echo "  __EFMigrationsHistory table does not exist"
  fi
}

migrate_and_or_seed_database_in_dotnet() {
  local db_name=$1
  local new_db=$2
  if ! command -v curl >/dev/null 2>&1; then
    echo "  ERROR: curl is not available!"
    return 1
  fi

  stop_services "yes-all-except-control-pyrunner"
  
  local seed=false
  if [[ "$new_db" == "true" ]] || [[ "$EXECUTE_DATABASE_OPS" == *"seed"* ]]; then
    seed=true
  fi

  local migrate=false
  if [[ "$new_db" == "true" ]]; then
    migrate=true
  else
    if [[ "$EXECUTE_DATABASE_OPS" == *"migrate"* ]]; then
      check_if_migration_table_exists $bodhi_db_name
      if [[ "$migration_table_exists" == "true" ]]; then
        migrate=true
      else
        echo "  ----------------------------------------------------------------------"  
        echo "  ERROR: database migration cannot be executed!"
        echo "  the database ${db_name} was not initialized by the dotnet application!"
        echo "  ----------------------------------------------------------------------" 

        return 1 
      fi
    fi
  fi

  local config_file_path='/bodhi/run/current/apps/bodhi-dotnet/api/appsettings.Production.json'

  if [[ "$seed" == "true" ]]; then
    echo "  disable authentication"
    local from='"Required": "true"'
    local to='"Required": "false"'
    sed -i "s|${from}|${to}|g" "${config_file_path}"

    echo "  enable seeding"
    from='"EnableSeeding": false'
    to='"EnableSeeding": true'
    sed -i "s|${from}|${to}|g" "${config_file_path}"
  fi

  if [[ "$migrate" == "true" ]]; then
    echo "  enable database migrations at startup"
    local from='"MigrateDatabase": false'
    local to='"MigrateDatabase": true'
    sed -i "s|${from}|${to}|g" "${config_file_path}"
  fi

  start_bodhi_apps_service

  # wait just a bit for the API startup to finish
  sleep 30

  if [[ "$migrate" == "true" ]]; then
    # just running the API service also DID the DB migration at startup
    echo "  ${db_name} database migration was executed at service startup"
  fi

  if [[ "$seed" == "true" ]]; then
    echo "  seed the database $db_name by calling the seed endpoints"
    local base_url="http://127.0.0.1:5041/web"
    local endpoints=(
      "/seed?seedMirrorData=false&seedOpticalFilterData=false&seedPlateTypesData=true&seedProtocolData=true"
    )

    for endpoint in "${endpoints[@]}"; do
      local url="${base_url/}${endpoint}"
      echo "   call seed endpoint at ${endpoint}..."
      local temp_file=$(mktemp)
      local retry_count=0
      local max_retries=3
      
      while [ $retry_count -lt $max_retries ]; do
        http_status=$(curl --request POST --url ${url} --silent --write-out "%{http_code}\n" --output $temp_file)
        response_body=$(cat $temp_file)
        
        # Check if status code is in 200s range
        if [[ "$http_status" =~ ^2[0-9][0-9]$ ]]; then
          echo "   response status: $http_status, body: $response_body"
          break
        else
          retry_count=$((retry_count + 1))
          if [ $retry_count -lt $max_retries ]; then
            echo "   response status: $http_status (attempt $retry_count/$max_retries), retrying in 5 seconds..."
            sleep 5
          else
            echo "   response status: $http_status, body: $response_body (failed after $max_retries attempts)"
          fi
        fi
      done
      
      rm -f $temp_file
    done
  fi

  stop_bodhi_apps_service

  if [[ "$seed" == "true" ]]; then
    echo "  re-enable authentication"
    local from='"Required": "false"'
    local to='"Required": "true"'
    sed -i "s|${from}|${to}|g" "${config_file_path}"

    echo "  disable seeding"
    from='"EnableSeeding": true'
    to='"EnableSeeding": false'
    sed -i "s|${from}|${to}|g" "${config_file_path}"
  fi

  if [[ "$migrate" == "true" ]]; then
    echo "  disable database migrations at startup"
    local from='"MigrateDatabase": true'
    local to='"MigrateDatabase": false'
    sed -i "s|${from}|${to}|g" "${config_file_path}"
  fi

}

execute_database_ops() {

  if [[ "$EXECUTE_DATABASE_OPS" == "nop" ]]; then
    return 0
  fi

  if ! command -v psql >/dev/null 2>&1; then
    echo "  ERROR: psql is not available!'"
    return 1
  fi

  initial_dir=$(pwd)
  cd /tmp

  echo ""
  echo "  ============================================="
  echo "  database operations to execute: $EXECUTE_DATABASE_OPS"
  echo "  ============================================="

  allow_current_user_to_run_as_postgres_without_a_password
  set_postgres_user_password

  if [[ "$EXECUTE_DATABASE_OPS" == "delete" ]] || [[ "$EXECUTE_DATABASE_OPS" == "recreate" ]] || [[ "$EXECUTE_DATABASE_OPS" == "reinitialize" ]]; then
    # delete the DBs first if exist
    delete_database $bodhi_db_name

    # DON't delete the IDP DB, we still need the users!!!
  fi

  if [[ "$EXECUTE_DATABASE_OPS" == "delete" ]]; then
    echo "  we're done, we only delete the DBs without recreating and/or initializing them"
  else

    # create DBs if do not exist

    create_database $bodhi_db_name
    local new_bodhi_db=$new_db

    create_database $bodhi_idp_db_name
    local new_bodhi_idp_db=$new_db
  
    if [[ "$EXECUTE_DATABASE_OPS" == "recreate" ]]; then
      echo "  we're done, we only recreate the DBs without inititializing them"
    else

      if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]] && \
        ( \
          [[ "$new_bodhi_db" == "true" ]] || [[ "$EXECUTE_DATABASE_OPS" == *"seed"* ]] \
        ); then
        # new DB in nodejs apps OR migrate OR seed
        # then seed
        seed_database_in_nodejs $bodhi_db_name $new_bodhi_db
      fi

      if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]] && \
        ( \
          [[ "$new_bodhi_db" == "true" ]] || \
          [[ "$EXECUTE_DATABASE_OPS" == *"migrate"* ]] || [[ "$EXECUTE_DATABASE_OPS" == *"seed"* ]] \
        ); then
        migrate_and_or_seed_database_in_dotnet $bodhi_db_name $new_bodhi_db
      fi
    fi
  fi

  echo "  database operations to execute: $EXECUTE_DATABASE_OPS: done"
  echo "  ========================================================="
  echo ""

  if [[ -d "${initial_dir}" ]]; then
    cd $initial_dir
  fi
}

generate_databases_info() {
    local output_folder=$1
    local output_file=$2

    echo "====================================================================" >> "${output_file}"
    echo "Postgresql database info" >> "${output_file}"
    echo "------------------------" >> "${output_file}"

    if command -v psql >/dev/null 2>&1; then
      sudo -u postgres psql -d "${bodhi_db_name}" -c "SELECT version();" >> "${output_file}"

      echo "" >> "${output_file}"
      sudo -u postgres psql -d "${bodhi_db_name}" -c "SELECT datname AS \"Database Name\",pg_size_pretty(pg_database_size(datname)) AS \"Size\",pg_catalog.pg_get_userbyid(datdba) AS \"Owner\",encoding AS \"Encoding\",datcollate AS \"Collate\",datctype AS \"Ctype\" FROM pg_database WHERE datistemplate = false;" >> "${output_file}"
      echo "" >> "${output_file}"
    else
      echo "postgresql does not seem to be installed" >> "${output_file}"
      echo "" >> "${output_file}"
    fi

    echo "" >> "${output_file}"
    echo "====================================================================" >> "${output_file}"
    echo "" >> "${output_file}"
}

dump_table_as_json() {
    local table_name=$1
    local where_clause=$2
    local output_file=$3
    
    # Build the SQL query
    local sql_query="SELECT * FROM ${table_name}"
    if [[ -n "$where_clause" ]]; then
        sql_query="${sql_query} WHERE ${where_clause}"
    fi
    
    # Use LC_ALL=C to ensure consistent locale behavior across all systems
    # output_file is already an absolute path, no need for realpath
    LC_ALL=C sudo -u postgres psql -d "${bodhi_db_name}" -c "\\copy (SELECT COALESCE(json_agg(row_to_json(t)), '[]'::json) FROM (${sql_query}) t) TO '${output_file}'" > /dev/null 2>&1
    local copy_result=$?
    
    # Fallback: if \copy fails, try regular COPY with C locale
    if [[ $copy_result -ne 0 ]]; then
        LC_ALL=C sudo -u postgres psql -d "${bodhi_db_name}" -c "COPY (SELECT COALESCE(json_agg(row_to_json(t)), '[]'::json) FROM (${sql_query}) t) TO '${output_file}';" > /dev/null 2>&1
        copy_result=$?
    fi
    
    return $copy_result
}

dump_protocols_from_database() {
    local filter_clause="$1"
    
    # Check if PostgreSQL is available and database exists first
    if ! command -v psql >/dev/null 2>&1; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR PostgreSQLNotInstalled"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if sed and awk are available (used for JSON parsing)
    if ! command -v sed >/dev/null 2>&1 || ! command -v awk >/dev/null 2>&1; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR SedAwkNotInstalled"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    local db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '${bodhi_db_name}'")
    if [[ -z "$db_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR DatabaseNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if protocol table exists
    local table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'protocol'")
    if [[ -z "$table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR ProtocolTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if plate_type table exists
    local plate_type_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'plate_type'")
    if [[ -z "$plate_type_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR PlateTypeTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if protocols_plate_maps table exists
    local protocols_plate_maps_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'protocols_plate_maps'")
    if [[ -z "$protocols_plate_maps_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR ProtocolsPlateMapsTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if plate_map table exists
    local plate_map_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'plate_map'")
    if [[ -z "$plate_map_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR PlateMapTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if calculation table exists
    local calculation_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'calculation'")
    if [[ -z "$calculation_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR CalculationTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if operation table exists
    local operation_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'operation'")
    if [[ -z "$operation_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR OperationTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if auto_export table exists
    local auto_export_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'auto_export'")
    if [[ -z "$auto_export_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR AutoExportTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # Check if sample_group table exists
    local sample_group_table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'sample_group'")
    if [[ -z "$sample_group_table_exists" ]]; then
        echo "DUMP_PROTOCOLS_EXIT_ERROR SampleGroupTableNotFound"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # remove all non-alphanumeric characters from the filename
    # to make it more usable by the sql COPY in dump_table_as_json
    local current_datetime=$(date +"%Y%m%d%H%M%S")
    
    local dump_protocols_file_name="dumpprotocols${current_datetime}.json"
    local dump_protocols_file_path="${tmp_bodhi_folder_dumps}/${dump_protocols_file_name}"
    
    local dump_plate_types_file_name="dumpplatetypes${current_datetime}.json"
    local dump_plate_types_file_path="${tmp_bodhi_folder_dumps}/${dump_plate_types_file_name}"
    
    local dump_protocols_plate_maps_file_name="dumpprotocolsplatemaps${current_datetime}.json"
    local dump_protocols_plate_maps_file_path="${tmp_bodhi_folder_dumps}/${dump_protocols_plate_maps_file_name}"
    
    local dump_plate_map_file_name="dumpplatemap${current_datetime}.json"
    local dump_plate_map_file_path="${tmp_bodhi_folder_dumps}/${dump_plate_map_file_name}"
    
    local dump_calculation_file_name="dumpcalculation${current_datetime}.json"
    local dump_calculation_file_path="${tmp_bodhi_folder_dumps}/${dump_calculation_file_name}"
    
    local dump_operation_file_name="dumpoperation${current_datetime}.json"
    local dump_operation_file_path="${tmp_bodhi_folder_dumps}/${dump_operation_file_name}"
    
    local dump_child_operation_file_name="dumpchildoperation${current_datetime}.json"
    local dump_child_operation_file_path="${tmp_bodhi_folder_dumps}/${dump_child_operation_file_name}"
    
    local dump_auto_export_file_name="dumpautoexport${current_datetime}.json"
    local dump_auto_export_file_path="${tmp_bodhi_folder_dumps}/${dump_auto_export_file_name}"
    
    local dump_sample_group_file_name="dumpsamplegroup${current_datetime}.json"
    local dump_sample_group_file_path="${tmp_bodhi_folder_dumps}/${dump_sample_group_file_name}"
    
    # Function to clean up temporary dump files
    cleanup_temp_dump_files() {
        rm -f "${dump_protocols_file_path}"
        rm -f "${dump_plate_types_file_path}"
        rm -f "${dump_protocols_plate_maps_file_path}"
        rm -f "${dump_plate_map_file_path}"
        rm -f "${dump_calculation_file_path}"
        rm -f "${dump_operation_file_path}"
        rm -f "${dump_child_operation_file_path}"
        rm -f "${dump_auto_export_file_path}"
        rm -f "${dump_sample_group_file_path}"
    }
    
    # Create dumps directory if it doesn't exist
    mkdir -p "${tmp_bodhi_folder_dumps}"
    chmod 777 "${tmp_bodhi_folder_dumps}"
    
    # Initialize all files with empty JSON arrays
    echo "[]" > "${dump_protocols_file_path}"
    echo "[]" > "${dump_plate_types_file_path}"
    echo "[]" > "${dump_protocols_plate_maps_file_path}"
    echo "[]" > "${dump_plate_map_file_path}"
    echo "[]" > "${dump_calculation_file_path}"
    echo "[]" > "${dump_operation_file_path}"
    echo "[]" > "${dump_child_operation_file_path}"
    echo "[]" > "${dump_auto_export_file_path}"
    echo "[]" > "${dump_sample_group_file_path}"
    
    # Set proper permissions (666 so postgres user can write to them)
    chmod 666 "${dump_protocols_file_path}"
    chmod 666 "${dump_plate_types_file_path}"
    chmod 666 "${dump_protocols_plate_maps_file_path}"
    chmod 666 "${dump_plate_map_file_path}"
    chmod 666 "${dump_calculation_file_path}"
    chmod 666 "${dump_operation_file_path}"
    chmod 666 "${dump_child_operation_file_path}"
    chmod 666 "${dump_auto_export_file_path}"
    chmod 666 "${dump_sample_group_file_path}"

    # Dump protocols table as JSON using the helper function
    # Only dump protocols where is_current = true AND deleted_at IS NULL AND apply any additional filter
    local protocol_filter_clause="is_current = true AND deleted_at IS NULL"
    if [[ -n "$filter_clause" ]]; then
        protocol_filter_clause="is_current = true AND deleted_at IS NULL AND (${filter_clause})"
    fi
    
    dump_table_as_json "protocol" "${protocol_filter_clause}" "${dump_protocols_file_path}"

    if [[ $? -ne 0 ]]; then
        cleanup_temp_dump_files
        echo "DUMP_PROTOCOLS_EXIT_ERROR FailedToDumpProtocols"
        echo "DUMP_PROTOCOLS_EXIT_STATUS false"
        return 1
    fi

    # NOTE: all this complex grep+sed+awk and line-by-line parsing 
    # is used to extract from json files 
    # ALL because 'jq' is not installed on target/instrument !!

    # Extract protocol id and version pairs from the protocol dump: this is the PK
    local protocol_pks=""
    
    # Use grep to extract id and version values, then combine them
    local ids=$(grep -o '"id"[[:space:]]*:[[:space:]]*"[^"]*"' "${dump_protocols_file_path}" | sed 's/.*"id"[[:space:]]*:[[:space:]]*"\([^"]*\)".*/\1/')
    local versions=$(grep -o '"version"[[:space:]]*:[[:space:]]*"[^"]*"' "${dump_protocols_file_path}" | sed 's/.*"version"[[:space:]]*:[[:space:]]*"\([^"]*\)".*/\1/')
    
    # Convert to arrays and combine them
    local id_array=($ids)
    local version_array=($versions)
    local tuple_list=""
    
    # Combine ids and versions into tuples
    for ((i=0; i<${#id_array[@]}; i++)); do
        if [[ -n "${id_array[i]}" ]] && [[ -n "${version_array[i]}" ]]; then
            if [[ -n "$tuple_list" ]]; then
                tuple_list="${tuple_list},('${id_array[i]}','${version_array[i]}')"
            else
                tuple_list="('${id_array[i]}','${version_array[i]}')"
            fi
        fi
    done
    
    protocol_pks="$tuple_list"

    if [[ -n "$protocol_pks" ]]; then

        # Extract plate_type values from the protocol dump and format them as quoted strings
        local plate_type_ids=$(grep -o '"plate_type"[[:space:]]*:[[:space:]]*"[^"]*"' "${dump_protocols_file_path}" | \
            sed 's/.*"plate_type"[[:space:]]*:[[:space:]]*"\([^"]*\)".*/\1/' | \
            grep -v '^null$' | sort -u | sed "s/^/'/; s/$/'/" | tr '\n' ',' | sed 's/,$//')

        if [[ -n "$plate_type_ids" ]]; then
            # Build WHERE clause for plate_type table
            local plate_type_where_clause="id IN (${plate_type_ids})"
            
            # Dump plate_type table as JSON
            dump_table_as_json "plate_type" "${plate_type_where_clause}" "${dump_plate_types_file_path}"
            
            if [[ $? -ne 0 ]]; then
                echo "Warning: Failed to dump plate types"
            fi
        else
            echo "WARNING: No plate_type references found in protocols ?"
        fi

        # Build WHERE clause for tables that reference protocol composite key
        local protocol_reference_where_clause="(protocol_id, protocol_version) IN (${protocol_pks})"
        
        # Dump protocols_plate_maps table as JSON
        dump_table_as_json "protocols_plate_maps" "${protocol_reference_where_clause}" "${dump_protocols_plate_maps_file_path}"
        
        if [[ $? -ne 0 ]]; then
            echo "Warning: Failed to dump protocols plate maps"
        fi
        
        # Dump calculation table as JSON
        dump_table_as_json "calculation" "${protocol_reference_where_clause}" "${dump_calculation_file_path}"
        
        if [[ $? -ne 0 ]]; then
            echo "Warning: Failed to dump calculations"
        fi
        
        # Dump operation table as JSON
        dump_table_as_json "operation" "${protocol_reference_where_clause}" "${dump_operation_file_path}"
        
        if [[ $? -ne 0 ]]; then
            echo "Warning: Failed to dump operations"
        fi

        # Extract id values from the operation dump for child operations
        local operation_ids=$(grep -o '"id"[[:space:]]*:[[:space:]]*[0-9]\+' "${dump_operation_file_path}" | \
            sed 's/.*"id"[[:space:]]*:[[:space:]]*\([0-9]\+\).*/\1/' | \
            sort -u | tr '\n' ',' | sed 's/,$//')

        if [[ -n "$operation_ids" ]]; then
            # Build WHERE clause for child operations using extracted operation_ids
            local child_operation_where_clause="parent_id IN (${operation_ids})"
            
            # Dump child operations to the new file
            dump_table_as_json "operation" "${child_operation_where_clause}" "${dump_child_operation_file_path}"
            
            if [[ $? -ne 0 ]]; then
                echo "Warning: Failed to dump child operations"
            else
                # Check if child operations file contains data (not just empty array)
                local child_operations_content=$(cat "${dump_child_operation_file_path}" | tr -d '\n' | sed 's/^[[:space:]]*\[\][[:space:]]*$//')
                if [[ -n "$child_operations_content" && "$child_operations_content" != "[]" ]]; then
                    # Merge child operations into the main operation file using temp files
                    local temp_merge_file=$(mktemp)
                    
                    # Remove the closing bracket from main operations and write to temp file
                    sed 's/\]$//' "${dump_operation_file_path}" > "${temp_merge_file}"
                    
                    # Add comma separator to temp file
                    echo -n "," >> "${temp_merge_file}"
                    
                    # Extract child operation items (without outer brackets) and append to temp file
                    sed 's/^\[//; s/\]$//' "${dump_child_operation_file_path}" >> "${temp_merge_file}"
                    
                    # Add closing bracket
                    echo "]" >> "${temp_merge_file}"
                    
                    # Replace the original file with merged content
                    mv "${temp_merge_file}" "${dump_operation_file_path}"
                fi
            fi
        fi
        
        # Dump auto_export table as JSON
        dump_table_as_json "auto_export" "${protocol_reference_where_clause}" "${dump_auto_export_file_path}"
        
        if [[ $? -ne 0 ]]; then
            echo "Warning: Failed to dump auto exports"
        fi

        # Extract plate_map_id values from the protocols_plate_maps dump
        local plate_map_ids=$(grep -o '"plate_map_id"[[:space:]]*:[[:space:]]*"[^"]*"' "${dump_protocols_plate_maps_file_path}" | \
            sed 's/.*"plate_map_id"[[:space:]]*:[[:space:]]*"\([^"]*\)".*/\1/' | \
            grep -v '^null$' | sort -u | sed "s/^/'/; s/$/'/" | tr '\n' ',' | sed 's/,$//')

        if [[ -n "$plate_map_ids" ]]; then
            # Build WHERE clause for plate_map table
            local plate_map_where_clause="id IN (${plate_map_ids})"
            
            # Dump plate_map table as JSON
            dump_table_as_json "plate_map" "${plate_map_where_clause}" "${dump_plate_map_file_path}"
            
            if [[ $? -ne 0 ]]; then
                echo "Warning: Failed to dump plate maps"
            fi

            # Build WHERE clause for sample_group table
            local plate_map_where_clause="plate_map_id IN (${plate_map_ids})"

            # Dump sample_group table using the same plate_map_ids
            dump_table_as_json "sample_group" "${plate_map_where_clause}" "${dump_sample_group_file_path}"
            
            if [[ $? -ne 0 ]]; then
                echo "Warning: Failed to dump sample groups"
            fi
        else
            echo "Warning: No plate_map_id references found in protocols_plate_maps ?"
        fi
    else
        echo "No protocols found in protocol table"
    fi
    
    # Create the final consolidated JSON file
    local current_datetime=$(date +"%Y-%m-%d_%H-%M-%S")
    local protocols_file_name="protocols_${current_datetime}.json"
    local protocols_file_path="${tmp_bodhi_folder_dumps}/${protocols_file_name}"
    {
        echo "{"
        echo "    \"deprecated_protocols\": [],"
        echo "    \"protocol\": $(cat "${dump_protocols_file_path}" | tr -d '\n'),"
        echo "    \"plate_type\": $(cat "${dump_plate_types_file_path}" | tr -d '\n'),"
        echo "    \"protocols_plate_maps\": $(cat "${dump_protocols_plate_maps_file_path}" | tr -d '\n'),"
        echo "    \"plate_map\": $(cat "${dump_plate_map_file_path}" | tr -d '\n'),"
        echo "    \"calculation\": $(cat "${dump_calculation_file_path}" | tr -d '\n'),"
        echo "    \"operation\": $(cat "${dump_operation_file_path}" | tr -d '\n'),"
        echo "    \"auto_export\": $(cat "${dump_auto_export_file_path}" | tr -d '\n'),"
        echo "    \"sample_group\": $(cat "${dump_sample_group_file_path}" | tr -d '\n')"
        echo "}"
    } > "${protocols_file_path}"
    if [[ $? -ne 0 ]]; then
        echo "Warning: Failed to create consolidated protocols file"
    fi

    # Clean up individual dump files, keeping only the consolidated file
    cleanup_temp_dump_files
    
    echo "DUMP_PROTOCOLS_EXIT_ERROR NoError"
    echo "DUMP_PROTOCOLS_EXIT_STATUS true"
    echo "DUMP_PROTOCOLS_FILE_NAME ${protocols_file_name}"
    echo "DUMP_PROTOCOLS_FILE_PATH ${protocols_file_path}"
}

update_protocols_in_database() {
    local filter_clause="$1"
    local update_statement="$2"
    
    echo ""
    echo "  ============================================="
    echo "  update protocols in database"
    echo "  filter: $filter_clause"
    echo "  update: $update_statement"
    echo ""
    
    # Check if PostgreSQL is available and database exists first
    if ! command -v psql >/dev/null 2>&1; then
        echo "  ERROR: PostgreSQL is not installed or psql command not available!"
        echo ""
        return 1
    fi

    # Validate input parameters
    if [[ -z "$filter_clause" ]]; then
        echo "  ERROR: Filter clause is required but was not provided!"
        echo ""
        return 1
    fi

    if [[ -z "$update_statement" ]]; then
        echo "  ERROR: Update statement is required but was not provided!"
        echo ""
        return 1
    fi

    local db_exists=$(sudo -u postgres psql -tc "SELECT 1 FROM pg_database WHERE datname = '${bodhi_db_name}'")
    if [[ -z "$db_exists" ]]; then
        echo "  ERROR: Database '${bodhi_db_name}' was not found!"
        echo ""
        return 1
    fi

    # Check if protocol table exists
    local table_exists=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT 1 FROM information_schema.tables WHERE table_name = 'protocol'")
    if [[ -z "$table_exists" ]]; then
        echo "  ERROR: Protocol table does not exist in database '${bodhi_db_name}'!"
        echo ""
        return 1
    fi

    # Validate that the filter clause would match existing records (safety check)
    local match_count=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "SELECT COUNT(*) FROM protocol WHERE (${filter_clause})" 2>/dev/null)
    local query_result=$?
    
    if [[ $query_result -ne 0 ]]; then
        echo "  ERROR: Invalid filter clause syntax: '${filter_clause}'"
        echo ""
        return 1
    fi

    match_count=$(echo "$match_count" | xargs)
    if [[ ! "$match_count" =~ ^[0-9]+$ ]]; then
        echo "  ERROR: Filter clause returned invalid result, expected a number but got: '${match_count}'"
        echo ""
        return 1
    fi

    if [[ "$match_count" -eq 0 ]]; then
        echo "  ERROR: No protocols match the filter clause: '${filter_clause}'"
        echo ""
        return 1
    fi

    # Test the update statement syntax by doing a dry run with LIMIT 0 (validates columns but updates no rows)
    local syntax_test=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "UPDATE protocol SET ${update_statement} WHERE (${filter_clause}) AND ctid IN (SELECT ctid FROM protocol LIMIT 0)" 2>&1)
    local syntax_result=$?
    
    # Check if the output contains error messages (psql might return 0 even with SQL errors)
    if [[ $syntax_result -ne 0 ]] || [[ "$syntax_test" == *"ERROR:"* ]] || [[ "$syntax_test" == *"error:"* ]]; then
        echo "  ERROR: Invalid update statement syntax: '${update_statement}'"
        echo "  SQL Error: ${syntax_test}"
        echo ""
        return 1
    fi

    # Execute the update
    local update_result=$(sudo -u postgres psql -d "${bodhi_db_name}" -tc "UPDATE protocol SET ${update_statement} WHERE (${filter_clause})" 2>&1)
    local update_exit_code=$?
    
    if [[ $update_exit_code -ne 0 ]]; then
        echo "  ERROR: Update operation failed: ${update_result}"
        echo ""
        return 1
    fi

    # Extract the number of updated rows from the UPDATE result
    local updated_count=$(echo "$update_result" | grep -o 'UPDATE [0-9]*' | grep -o '[0-9]*')
    if [[ ! "$updated_count" =~ ^[0-9]+$ ]]; then
        updated_count="0"
    fi

    # Verify the update was successful by checking if the expected number of rows were affected
    if [[ "$updated_count" -ne "$match_count" ]]; then
        echo "  ERROR: Expected to update $match_count protocols, but actually updated $updated_count"
    else
        echo "  Successfully updated $updated_count protocols"
    fi

    echo ""
    echo "  update protocols in database: done"
    echo "  =========================================="
    echo ""
}
