# ====================================================================
# PyRunner Management Functions
# ====================================================================
# This file contains functions for managing PyRunner installations.
# All paths and configuration are provided by the caller for complete flexibility.
# ====================================================================

# Create PyRunner installation report and diff analysis
# Parameters (named, passed as key=value):
#   stage=string (required - stage for report files: "initial" or "final")
#   reports_dir=path (required - path where reports will be stored)
#   home_dir=path (required - path to PyRunner home directory to analyze)
create_pyrunner_report() {
    # Initialize default values
    local stage=""
    local reports_dir=""
    local home_dir=""

    # Parse named parameters
    for param in "$@"; do
        case "$param" in
            stage=*)
                stage="${param#stage=}"
                ;;
            reports_dir=*)
                reports_dir="${param#reports_dir=}"
                ;;
            home_dir=*)
                home_dir="${param#home_dir=}"
                ;;
            *)
                echo "   WARNING: unknown parameter: $param (ignoring)"
                ;;
        esac
    done
    
    # Validate required parameters
    if [[ -z "$stage" ]]; then
        echo "   ERROR: stage parameter is required!"
        return 1
    fi
    if [[ -z "$reports_dir" ]]; then
        echo "   ERROR: reports_dir parameter is required!"
        return 1
    fi
    if [[ -z "$home_dir" ]]; then
        echo "   ERROR: home_dir parameter is required!"
        return 1
    fi
    
    local report_file="$reports_dir/${stage}.txt"

    mkdir -p "$reports_dir"
    rm -f "$report_file"
    touch "$report_file"
    chmod 644 "$report_file"

    {
        echo "   PyRunner installation report"
        echo "   ============================="
        echo ""
        echo "   Directory structure in $home_dir:"
        echo "   =================================="
        echo ""
        echo "   All directories (recursive):"
        find "$home_dir" -type d 2>/dev/null | sort
        echo ""
        echo "   All files:"
        find "$home_dir" -type f 2>/dev/null | sort

        echo ""
        echo "   Systemd service files:"
        echo "   ========================="
        find /etc/systemd/system/ -type f 2>/dev/null | sort

        echo ""
        echo "   Profile.d scripts:"
        echo "   ==================="
        find /etc/profile.d/ -type f 2>/dev/null | sort

    } >> "$report_file"

    files_ext_to_diff=(
        "ini"
        "iss"
        "cfg"
        "conf"
        "config"
        "yml"
        "yaml"
        "json"
        "txt"
        "log"
        "sh"
        "service"
        "srv"
        "ctl"
    )

    copy_dir="$reports_dir/files/${stage}"
    rm -rf "$copy_dir"
    mkdir -p "$copy_dir"

    for ext in "${files_ext_to_diff[@]}"; do
        find "$home_dir" -type f -name "*.$ext" -exec cp --parents {} "$copy_dir" \;
        find /etc/systemd/system/ -type f -name "*.$ext" -exec sh -c 'cp --parents "$1" "$2" && chown --reference="$2" "$2/$1" && chmod o+r "$2/$1"' _ {} "$copy_dir" \;
        find /etc/profile.d/ -type f -name "*.$ext" -exec sh -c 'cp --parents "$1" "$2" && chown --reference="$2" "$2/$1" && chmod o+r "$2/$1"' _ {} "$copy_dir" \;
    done

    if [[ "$stage" == "final" ]]; then
        initial_files="$reports_dir/files/initial"
        final_files="$reports_dir/files/final"
        diff_report="$reports_dir/diff_report.txt"

        {
            echo "   File Comparison Report (Initial vs Final)"
            echo "   ========================================"
            echo ""
        } > "$diff_report"

        # Initialize counters
        local count_new=0
        local count_removed=0
        local count_identical=0
        local count_modified=0
        local count_error=0

        # Arrays to store files by category
        local files_identical=()
        local files_modified=()
        local files_new=()
        local files_removed=()
        local files_error=()

        # Get all files from both directories
        initial_file_list=$(find "$initial_files" -type f 2>/dev/null | sed "s|^$initial_files||" | sort)
        final_file_list=$(find "$final_files" -type f 2>/dev/null | sed "s|^$final_files||" | sort)
        
        # Combine and get unique file paths
        all_files=$(echo -e "$initial_file_list\n$final_file_list" | sort -u)

        # First pass: categorize all files
        for relative_file in $all_files; do
            initial_file="$initial_files$relative_file"
            final_file="$final_files$relative_file"

            if [[ ! -f "$initial_file" ]]; then
                files_new+=("$relative_file")
                ((count_new++))
            elif [[ ! -f "$final_file" ]]; then
                files_removed+=("$relative_file")
                ((count_removed++))
            elif cmp -s "$initial_file" "$final_file"; then
                files_identical+=("$relative_file")
                ((count_identical++))
            else
                # Check if diff can be generated
                diff -u --strip-trailing-cr "$initial_file" "$final_file" >/dev/null 2>&1
                local diff_exit_code=$?
                if [[ $diff_exit_code -eq 0 ]]; then
                    # Exit code 0 means files are identical [ with --strip-trailing-cr ]
                    files_identical+=("$relative_file")
                    ((count_identical++))
                elif [[ $diff_exit_code -eq 1 ]]; then
                    # Exit code 1 means files differ (normal case)
                    files_modified+=("$relative_file")
                    ((count_modified++))
                elif [[ $diff_exit_code -eq 2 ]]; then
                    # Exit code 2 means error occurred
                    files_error+=("$relative_file")
                    ((count_error++))
                else
                    # unknown exit code, treat as error
                    files_error+=("$relative_file")
                    ((count_error++))
                fi
            fi
        done

        # Second pass: write grouped results
        
        # Group 1: INFO - Identical files
        if [[ ${#files_identical[@]} -gt 0 ]]; then
            {
                echo "   INFO: IDENTICAL FILES ($count_identical files)"
                echo "   INFO: File(s) are identical between initial and final state"
                echo "   ============================================="
                echo ""
            } >> "$diff_report"
            
            for relative_file in "${files_identical[@]}"; do
                echo "   $relative_file" >> "$diff_report"
            done

            echo "   =============================================" >> "$diff_report"
            echo "" >> "$diff_report"
        fi

        # Group 2: DIFF - Modified files
        if [[ ${#files_modified[@]} -gt 0 ]]; then
            {
                echo "   DIFF: MODIFIED FILES ($count_modified files)"
                echo "   DIFF: File(s) differ between initial and final state"
                echo "   ============================================"
                echo ""
            } >> "$diff_report"
            
            for relative_file in "${files_modified[@]}"; do
                initial_file="$initial_files$relative_file"
                final_file="$final_files$relative_file"
                
                {
                    echo "   $relative_file"
                    echo "   ----------------------------------------"
                } >> "$diff_report"
                diff -u --strip-trailing-cr "$initial_file" "$final_file" >> "$diff_report" 2>/dev/null
                echo "" >> "$diff_report"
            done

            echo "   ============================================" >> "$diff_report"
            echo "" >> "$diff_report"
        fi

        # Group 3: ERROR - New files (created)
        if [[ ${#files_new[@]} -gt 0 ]]; then
            {
                echo "   ERROR: NEW FILES CREATED ($count_new files)"
                echo "   ERROR: File(s) only exists in final state (new file created)"
                echo "   ==========================================="
                echo ""
            } >> "$diff_report"
            
            for relative_file in "${files_new[@]}"; do
                echo "   $relative_file" >> "$diff_report"
            done

            echo "" >> "$diff_report"
            echo "   ============================================" >> "$diff_report"
            echo "   END ERROR: NEW FILES CREATED ($count_new files)" >> "$diff_report"
            echo "" >> "$diff_report"
        fi

        # Group 4: ERROR - Removed files
        if [[ ${#files_removed[@]} -gt 0 ]]; then
            {
                echo "   ERROR: REMOVED FILES ($count_removed files)"
                echo "   ERROR: File(s) only exists in initial state (file was removed)"
                echo "   ==========================================="
                echo ""
            } >> "$diff_report"
            
            for relative_file in "${files_removed[@]}"; do
                echo "   $relative_file" >> "$diff_report"
            done

            echo "   ============================================" >> "$diff_report"
            echo "" >> "$diff_report"
        fi

        # Group 5: ERROR - Diff errors
        if [[ ${#files_error[@]} -gt 0 ]]; then
            {
                echo "   ERROR: DIFF PROCESSING ERRORS ($count_error files)"
                echo "   ERROR: Could not generate diff or unexpected diff result"
                echo "   =================================================="
                echo ""
            } >> "$diff_report"
            
            for relative_file in "${files_error[@]}"; do
                echo "   $relative_file" >> "$diff_report"
            done

            echo "   ==================================================" >> "$diff_report"
            echo "" >> "$diff_report"
        fi

        # Write summary
        {
            echo ""
            echo "   SUMMARY"
            echo "   ======="
            echo "   Files created (new):     $count_new"
            echo "   Files removed:           $count_removed"
            echo "   Files identical:         $count_identical"
            echo "   Files modified:          $count_modified"
            echo "   Diff errors:             $count_error"
            echo "   --------------------------------"
            echo "   Total files processed:   $((count_new + count_removed + count_identical + count_modified + count_error))"
        } >> "$diff_report"

        rm -rf $initial_files
        rm -rf $final_files

        echo "  PyRunner diff report saved to: $diff_report"
    fi
    
    return 0
}

# Validate PyRunner installation structure and files
# Parameters (named, passed as key=value):
#   source_install_config_file=path (required - path to install config file)
#   source_firmware=path (required - path to firmware directory)
#   source_protocols=path (optional - path to protocols directory)
#   source_pypack=path (required - path to PyPack directory)
#   source_raspbian=path (required - path to Raspbian directory)
#   source_config=path (optional - path to config directory)
validate_pyrunner() {
    # Initialize default values
    local source_install_config_file=""
    local source_firmware=""
    local source_protocols=""
    local source_pypack=""
    local source_raspbian=""
    local source_config=""

    # Parse named parameters
    for param in "$@"; do
        case "$param" in
            source_install_config_file=*)
                source_install_config_file="${param#source_install_config_file=}"
                ;;
            source_firmware=*)
                source_firmware="${param#source_firmware=}"
                ;;
            source_protocols=*)
                source_protocols="${param#source_protocols=}"
                ;;
            source_pypack=*)
                source_pypack="${param#source_pypack=}"
                ;;
            source_raspbian=*)
                source_raspbian="${param#source_raspbian=}"
                ;;
            source_config=*)
                source_config="${param#source_config=}"
                ;;
            *)
                echo "   WARNING: unknown parameter: $param (ignoring)"
                ;;
        esac
    done
    
    # Validate required parameters
    if [[ -z "$source_install_config_file" ]]; then
        echo "   ERROR: source_install_config_file parameter is required!"
        return 1
    fi
    if [[ -z "$source_firmware" ]]; then
        echo "   ERROR: source_firmware parameter is required!"
        return 1
    fi
    if [[ -z "$source_pypack" ]]; then
        echo "   ERROR: source_pypack parameter is required!"
        return 1
    fi
    if [[ -z "$source_raspbian" ]]; then
        echo "   ERROR: source_raspbian parameter is required!"
        return 1
    fi
    
    if [[ ! -f "$source_install_config_file" ]]; then
        # bail out if we can't read the setup variables
        # it means our assumptions about the install structure are wrong!!
        # and we cannot install the PyRunner correctly
        echo "   ERROR: required install config file does not exist: $source_install_config_file"
        echo "   ERROR: Installation structure is invalid - cannot proceed!"
        return 1
    fi

    # let's do some validations
    local required_dirs=(
        "$source_firmware"
        "$source_pypack"
        "$source_raspbian"
    )
    
    local optional_dirs=(
        "$source_protocols"
        "$source_config"
    )
    
    # Check required directories - bail out if any are missing
    for dir in "${required_dirs[@]}"; do
        if [[ ! -d "$dir" ]]; then
            echo "   ERROR: required source directory does not exist: $dir"
            echo "   ERROR: installation structure is invalid - cannot proceed!"
            return 1
        fi
    done
    
    # Check optional directories - warn but continue if missing
    for dir in "${optional_dirs[@]}"; do
        if [[ ! -d "$dir" ]]; then
            echo "  optional source directory does not exist: $dir"
        fi
    done

    # Validate required files in source_pypack
    required_files=(
        "$source_pypack/chkwho.sh"
        "$source_pypack/FWUpdate.sh"
        "$source_pypack/install.sh"
        "$source_pypack/installsystemd.sh"
        "$source_pypack/RestartPyRunner.sh"
        "$source_pypack/StartPyRunner.sh"
        "$source_pypack/StopPyRunner.sh"
        "$source_pypack/uninstall.sh"
        "$source_pypack/chownwdog.service"
        "$source_pypack/django.service"
        "$source_pypack/pyrunner.service"
    )

    # Optional files that can be missing but should be checked
    optional_files=(
        "$source_pypack/TermPy.sh"
        "$source_pypack/use_dhcp.sh"
        "$source_pypack/merge_fi_bottom_to_gc.sh"
    )

    for file in "${required_files[@]}"; do
        if [[ ! -f "$file" ]]; then
            echo "   ERROR: required file does not exist: $file"
            echo "   ERROR: installation structure is invalid - cannot proceed!"
            return 1
        fi
    done

    for file in "${optional_files[@]}"; do
        if [[ ! -f "$file" ]]; then
            echo "  optional file does not exist: $file"
        fi
    done

    # Validate required wheel files in source_pypack 
    local source_fwwriter_wheel=$(find "$source_pypack" -name "fwwriter-*.whl" -type f 2>/dev/null | head -1)
    local source_pyrunner_wheel=$(find "$source_pypack" -name "pyrunner-*.whl" -type f 2>/dev/null | head -1)
    
    if [[ -z "$source_fwwriter_wheel" ]]; then
        echo "   ERROR: required fwwriter wheel file not found in $source_pypack"
        echo "   ERROR: expected pattern: fwwriter-*.whl (e.g., fwwriter-1.0.3092.14-py3-none-any.whl)"
        echo "   ERROR: installation structure is invalid - cannot proceed!"
        return 1
    fi
    
    if [[ -z "$source_pyrunner_wheel" ]]; then
        echo "   ERROR: required pyrunner wheel file not found in $source_pypack"
        echo "   ERROR: expected pattern: pyrunner-*.whl (e.g., pyrunner-1.1.3091.4-py2.py3-none-any.whl)"
        echo "   ERROR: installation structure is invalid - cannot proceed!"
        return 1
    fi

    # Read Setup variables from install config file if available
    local app_version=""
    local app_version_name=""
    local install_file_name=""
    
    if [[ -f "$source_install_config_file" ]]; then
        local app_version=$(get_ini_value "AppVersion" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
        if [[ -n "$app_version" ]]; then
            # Trim whitespace from start and end
            app_version=$(echo "$app_version" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        else
            echo "   ERROR: could not read AppVersion from install config file!"
        fi
        
        # Extract app_version_name from AppVerName and remove {code:GetDeviceIP}
        local app_ver_name=$(get_ini_value "AppVerName" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
        if [[ -n "$app_ver_name" ]]; then
            # Remove {code:GetDeviceIP} or any {code:...} pattern at the end and trim whitespace
            app_version_name=$(echo "$app_ver_name" | sed 's/[[:space:]]*{code:[^}]*}[[:space:]]*$//' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        else
            echo "   ERROR: could not read AppVerName from install config file!"
        fi
        
        install_file_name=$(get_ini_value "OutputBaseFilename" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
        if [[ -z "$install_file_name" ]]; then
            echo "   ERROR: could not read OutputBaseFilename from install config file!"
        fi
    else
        echo "   ERROR: install config file not found at $source_install_config_file"
    fi

    # Check that app_version and app_version_name are not empty
    if [[ -z "$app_version" ]] || [[ -z "$app_version_name" ]]; then
        echo "   ERROR: required version information missing!"
        echo "   ERROR: app_version='$app_version', app_version_name='$app_version_name'"
        echo "   ERROR: installation aborted - cannot proceed without version information."
        return 1
    else
        echo " validating PyRunner version to install..."
        echo "   app_version=$app_version"
        echo "   app_version_name=$app_version_name"
        echo "   install_file_name=$install_file_name"
    fi
}

# Backup existing PyRunner installation
# Parameters (named, passed as key=value):
#   current_path=path (required - path to current installation directory)
#   backup_path=path (required - path where backup will be stored)
backup_pyrunner() {
    # Initialize default values
    local current_path=""
    local backup_path=""
    
    # Parse named parameters
    for param in "$@"; do
        case "$param" in
            current_path=*)
                current_path="${param#current_path=}"
                ;;
            backup_path=*)
                backup_path="${param#backup_path=}"
                ;;
            *)
                echo "   WARNING: unknown parameter: $param (ignoring)"
                ;;
        esac
    done
    
    # Validate required parameters
    if [[ -z "$current_path" ]]; then
        echo "   ERROR: currentPath parameter is required!"
        return 1
    fi
    if [[ -z "$backup_path" ]]; then
        echo "   ERROR: backupPath parameter is required!"
        return 1
    fi
    
    # Validate that current directory exists
    if [[ ! -d "$current_path" ]]; then
        echo "   ERROR: current directory '$current_path' does not exist!"
        return 1
    fi
    
    echo "  backing up existing PyRunner..."
    sudo rm -rf "$backup_path"
    sudo mkdir -p "$backup_path"
    
    # Move existing directories to backup if they exist
    local directories_to_backup=(
        "$current_path/temp"
        "$current_path/logs"
        "$current_path/.config/pyrunner"
        "$current_path/.PyRInstall"
        "$current_path/pyrunner"
        "$current_path/hardware_scripts"
    )

    # Move existing files to backup if they exist
    local files_to_backup=(
        "$current_path/chkwho.sh"
        "$current_path/FWUpdate.sh"
        "$current_path/install.sh"
        "$current_path/installsystemd.sh"
        "$current_path/RestartPyRunner.sh"
        "$current_path/StartPyRunner.sh"
        "$current_path/StopPyRunner.sh"
        "$current_path/TermPy.sh"
        "$current_path/uninstall.sh"
        "$current_path/use_dhcp.sh"
        "$current_path/chownwdog.service"
        "$current_path/django.service"
        "$current_path/pyrunner.service"
        "$current_path/merge_fi_bottom_to_gc.sh"
    )

    # Backup directories preserving directory structure
    for directory in "${directories_to_backup[@]}"; do
        if [[ -d "$directory" ]]; then
            # Get relative path from current_path
            relative_path="${directory#$current_path/}"
            # Create parent directory structure in backup
            backup_parent_dir="$backup_path/$(dirname "$relative_path")"
            sudo mkdir -p "$backup_parent_dir"
            # Move preserving path structure
            sudo mv "$directory" "$backup_path/$relative_path"
        fi
    done

    # Backup files
    for file in "${files_to_backup[@]}"; do
        if [[ -f "$file" ]]; then
            file_name=$(basename "$file")
            sudo mv "$file" "$backup_path/$file_name"
        fi
    done
    
    return 0
}

# Perform the actual PyRunner installation
# Parameters (all named, passed as key=value):
#   device_ip=ip_address as string
#   python_version=python${major.minor} (required, e.g., python3.11)
#   target_user=username
#   target_user_group=groupname
#   inventory=inventory_string
#   is_simulation=true/false
#   is_develop=true/false
#   is_prototyping=true/false
#   flash_fw_flags=flags as string
#   force_flash_fw_flags=flags as string
#   current_pyrunner_pack_config_HAL=path (required - path to current HAL config file)
#   current_pyrunner_pack_config_MeasurementUnit=path (required - path to current MeasurementUnit config file)
#   source_install_config_file=path (required - path to install config file)
#   source_firmware=path (required - path to firmware directory)
#   source_protocols=path (optional - path to protocols directory)
#   source_pypack=path (required - path to PyPack directory)
#   source_raspbian=path (required - path to Raspbian directory)
#   source_config=path (optional - path to config directory)
#   source_prototypes=path (optional - path to prototypes directory)
#   pyrunner_systemd=path (required)
#   django_systemd=path (required)
#   chownwdog_systemd=path (required)
perform_pyrunner_installation() {
    # Initialize default values
    local device_ip=""
    local python_version=""
    local target_user=""
    local target_user_group=""
    local inventory=""
    local is_simulation=false
    local is_develop=false
    local is_prototyping=false
    local flash_fw_flags=""
    local force_flash_fw_flags=""
    local current_pyrunner_pack_config_HAL=""
    local current_pyrunner_pack_config_MeasurementUnit=""
    local source_install_config_file=""
    local source_firmware=""
    local source_protocols=""
    local source_pypack=""
    local source_raspbian=""
    local source_config=""
    local source_prototypes=""
    local pyrunner_systemd=""
    local django_systemd=""
    local chownwdog_systemd=""
    
    # Parse named parameters
    for param in "$@"; do
        case "$param" in
            device_ip=*)
                device_ip="${param#device_ip=}"
                ;;
            python_version=*)
                python_version="${param#python_version=}"
                ;;  
            target_user=*)
                target_user="${param#target_user=}"
                ;;
            target_user_group=*)
                target_user_group="${param#target_user_group=}"
                ;;
            inventory=*)
                inventory="${param#inventory=}"
                ;;
            is_simulation=*)
                is_simulation="${param#is_simulation=}"
                ;;
            is_develop=*)
                is_develop="${param#is_develop=}"
                ;;
            is_prototyping=*)
                is_prototyping="${param#is_prototyping=}"
                ;;
            flash_fw_flags=*)
                flash_fw_flags="${param#flash_fw_flags=}"
                ;;
            force_flash_fw_flags=*)
                force_flash_fw_flags="${param#force_flash_fw_flags=}"
                ;;
            current_pyrunner_pack_config_HAL=*)
                current_pyrunner_pack_config_HAL="${param#current_pyrunner_pack_config_HAL=}"
                ;;
            current_pyrunner_pack_config_MeasurementUnit=*)
                current_pyrunner_pack_config_MeasurementUnit="${param#current_pyrunner_pack_config_MeasurementUnit=}"
                ;;
            source_install_config_file=*)
                source_install_config_file="${param#source_install_config_file=}"
                ;;
            source_firmware=*)
                source_firmware="${param#source_firmware=}"
                ;;
            source_protocols=*)
                source_protocols="${param#source_protocols=}"
                ;;
            source_pypack=*)
                source_pypack="${param#source_pypack=}"
                ;;
            source_raspbian=*)
                source_raspbian="${param#source_raspbian=}"
                ;;
            source_config=*)
                source_config="${param#source_config=}"
                ;;
            source_prototypes=*)
                source_prototypes="${param#source_prototypes=}"
                ;;
            pyrunner_systemd=*)
                pyrunner_systemd="${param#pyrunner_systemd=}"
                ;;
            django_systemd=*)
                django_systemd="${param#django_systemd=}"
                ;;
            chownwdog_systemd=*)
                chownwdog_systemd="${param#chownwdog_systemd=}"
                ;;
            *)
                echo "   WARNING: unknown parameter: $param (ignoring)"
                ;;
        esac
    done
    
    # Validate required parameters
    if [[ -z "$python_version" ]]; then
        echo "   ERROR: python_version parameter is required!"
        return 1
    fi
    if [[ -z "$target_user" ]]; then
        echo "   ERROR: targetUser parameter is required!"
        return 1
    fi
    if [[ -z "$target_user_group" ]]; then
        echo "   ERROR: targetUserGroup parameter is required!"
        return 1
    fi
    if [[ -z "$current_pyrunner_pack_config_HAL" ]]; then
        echo "   ERROR: current_pyrunner_pack_config_HAL parameter is required!"
        return 1
    fi
    if [[ -z "$current_pyrunner_pack_config_MeasurementUnit" ]]; then
        echo "   ERROR: current_pyrunner_pack_config_MeasurementUnit parameter is required!"
        return 1
    fi
    if [[ -z "$source_install_config_file" ]]; then
        echo "   ERROR: source_install_config_file parameter is required!"
        return 1
    fi
    if [[ -z "$source_firmware" ]]; then
        echo "   ERROR: source_firmware parameter is required!"
        return 1
    fi
    if [[ -z "$source_pypack" ]]; then
        echo "   ERROR: source_pypack parameter is required!"
        return 1
    fi
    if [[ -z "$source_raspbian" ]]; then
        echo "   ERROR: source_raspbian parameter is required!"
        return 1
    fi
    if [[ -z "$source_config" ]]; then
        echo "   ERROR: source_config parameter is required!"
        return 1
    fi
    if [[ -z "$pyrunner_systemd" ]]; then
        echo "   ERROR: pyrunner_systemd parameter is required!"
        return 1
    fi
    if [[ -z "$django_systemd" ]]; then
        echo "   ERROR: django_systemd parameter is required!"
        return 1
    fi
    if [[ -z "$chownwdog_systemd" ]]; then
        echo "   ERROR: chownwdog_systemd parameter is required!"
        return 1
    fi

    # Validate user exists
    if ! id "$target_user" &>/dev/null; then
        echo "   ERROR: user '$target_user' does not exist!"
        return 1
    fi

    # validate user is member of pyrunner_user_group
    if ! groups "$pyrunner_user" | grep -q "\b$pyrunner_user_group\b"; then
        echo "   ERROR: user '$pyrunner_user' is not a member of group '$pyrunner_user_group', cannot install pyrunner!"
        return 1
    fi  

    local home_dir="/home/${target_user}"
  
    # source: contains the files from the install that are next to run

    # destination: in home directory where the PyRunner runs    
    local home_dest_pyrinstall="${home_dir}/.PyRInstall"
    local home_dest_env="${home_dest_pyrinstall}/InstEnv"
    local home_dest_mots="${home_dest_pyrinstall}/mots"
    local home_dest_protocols="${home_dest_pyrinstall}/PresetProtocols"
    local home_dest_prototypes="${home_dest_pyrinstall}/Prototypes"
    local home_dest_temp="${home_dir}/temp"
    local home_dest_logs="${home_dir}/logs"
    local home_dest_config_pyrunner="${home_dir}/.config/pyrunner"
    local home_dest_pyrunner="${home_dir}/pyrunner"
    local home_dest_hardware_scripts="${home_dir}/hardware_scripts"

    local dest_pyrunner_pack_dir="${home_dest_pyrunner}/lib/${python_version}/site-packages/PyRunner"

    # Define destination config files
    local dest_pyrunner_pack_config="${dest_pyrunner_pack_dir}/config"
    local dest_pyrunner_pack_config_HAL="${dest_pyrunner_pack_config}/HAL.cfg"
    local dest_pyrunner_pack_config_ScanTable="${dest_pyrunner_pack_config}/ScanTable.cfg"
    local dest_pyrunner_pack_config_MeasurementUnit="${dest_pyrunner_pack_config}/MeasurementUnit.cfg"
    local dest_pyrunner_pack_config_TemperatureServices="${dest_pyrunner_pack_config}/TemperatureServices.cfg"
    local dest_pyrunner_pack_config_logger_server="${dest_pyrunner_pack_config}/log/logger_server.yaml"
    local dest_pyrunner_pack_config_logger="${dest_pyrunner_pack_config}/log/logger.yaml"
    local dest_pyrunner_pack_config_django="${dest_pyrunner_pack_dir}/webinterface/settings.cfg"

    # Read version info from install script
    local app_version=""
    local app_version_name=""
    local install_file_name=""
    
    if [[ -f "$source_install_config_file" ]]; then
        local app_version=$(get_ini_value "AppVersion" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
        if [[ -n "$app_version" ]]; then
            app_version=$(echo "$app_version" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        fi
        
        local app_ver_name=$(get_ini_value "AppVerName" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
        if [[ -n "$app_ver_name" ]]; then
            app_version_name=$(echo "$app_ver_name" | sed 's/[[:space:]]*{code:[^}]*}[[:space:]]*$//' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        fi
        
        install_file_name=$(get_ini_value "OutputBaseFilename" "Setup" "$source_install_config_file" 2>/dev/null || echo "")
    fi

    echo "  installing PyRunner..."
    echo "   PyRunner version to install:"
    echo "     app_version=$app_version"
    echo "     app_version_name=$app_version_name"
    echo "     install_file_name=$install_file_name"
    
    # Create directory structure
    echo "   creating directory structure..."
    sudo -u "$target_user" mkdir -p "$home_dest_temp"
    sudo -u "$target_user" mkdir -p "$home_dest_logs"
    sudo -u "$target_user" mkdir -p "$home_dest_logs/reports"
    sudo -u "$target_user" mkdir -p "$home_dest_config_pyrunner"
    sudo -u "$target_user" mkdir -p "$home_dest_config_pyrunner/simulation"
    sudo -u "$target_user" mkdir -p "$home_dest_env"
    sudo -u "$target_user" mkdir -p "$home_dest_mots"
    sudo -u "$target_user" mkdir -p "$home_dest_protocols"
    sudo -u "$target_user" mkdir -p "$home_dest_prototypes"
    sudo -u "$target_user" mkdir -p "$home_dest_hardware_scripts"
    
    # Set correct ownership for all created directories
    sudo chown -R "$target_user:$target_user_group" "$home_dest_temp"
    sudo chown -R "$target_user:$target_user_group" "$home_dest_logs"
    sudo chown -R "$target_user:$target_user_group" "$home_dest_config_pyrunner"
    sudo chown -R "$target_user:$target_user_group" "$home_dest_pyrinstall"
    sudo chown -R "$target_user:$target_user_group" "$home_dest_hardware_scripts"

    # Move Firmware files
    echo "   moving firmware files..."
    if [[ -d "$source_firmware" ]]; then
        sudo mv "$source_firmware/"* "$home_dest_mots/"
        sudo chown -R "$target_user:$target_user_group" "$home_dest_mots/"
    else
        echo "   WARNING: no firmware directory found at $source_firmware"
    fi

    # Move Preset Protocols (if they exist)
    echo "   moving preset protocols..."
    if [[ -d "$source_protocols" ]]; then
        sudo mv "$source_protocols/"* "$home_dest_protocols/"
        sudo chown -R "$target_user:$target_user_group" "$home_dest_protocols/"
    else
        echo "   WARNING: no preset protocols directory found at $source_protocols"
    fi

    # install Prototypes 
    if [[ "$is_prototyping" == "true" ]]; then
        # copy the prototypes to home_dest_hardware_scripts
        echo "   copying hardware scripts (prototypes)..."
        if [[ -d "$source_prototypes" ]]; then
            sudo cp -r "$source_prototypes/"* "$home_dest_hardware_scripts/"
            sudo chown -R "$target_user:$target_user_group" "$home_dest_hardware_scripts/"
        else
            echo "   WARNING: no prototypes directory found at $source_prototypes"
        fi

        echo "   moving prototypes..."
        if [[ -d "$source_prototypes" ]]; then
            sudo mv "$source_prototypes/"* "$home_dest_prototypes/"
            sudo chown -R "$target_user:$target_user_group" "$home_dest_prototypes/"
        else
            echo "   WARNING: no prototypes directory found at $source_prototypes"
        fi
    else
        rm -rf "$home_dest_prototypes"
        rm -rf "$home_dest_hardware_scripts"
    fi

    # Move Python packages
    echo "   moving Python packages..."
    if [[ -d "$source_raspbian" ]]; then
        sudo -u "$target_user" mkdir -p "$home_dest_env/Raspbian"
        sudo mv "$source_raspbian/"* "$home_dest_env/Raspbian/"
        sudo chown -R "$target_user:$target_user_group" "$home_dest_env/Raspbian/"
    else
        echo "   WARNING: no Raspbian packages found at $source_raspbian"
    fi

    # Move individual wheel files
    echo "   moving wheel files..."
    for wheel in "$source_pypack/"*.whl; do
        if [[ -f "$wheel" ]]; then
            sudo mv "$wheel" "$home_dest_env/"
            sudo chown "$target_user:$target_user_group" "$home_dest_env/$(basename "$wheel")"
        fi
    done

    # Move and set up shell scripts
    echo "   setting up shell scripts..."
    for script_file in "$source_pypack/"*.sh; do
        if [[ -f "$script_file" ]]; then
            script_name=$(basename "$script_file")
            sudo sed 's/\r$//' "$script_file" > "$home_dir/$script_name"
            sudo chmod 744 "$home_dir/$script_name"
            sudo chown "$target_user:$target_user_group" "$home_dir/$script_name"
            sudo rm "$script_file"
        fi
    done

    # Move systemd service files
    echo "   setting up systemd service files..."
    for service in pyrunner.service django.service chownwdog.service; do
        if [[ -f "$source_pypack/$service" ]]; then
            sudo mv "$source_pypack/$service" "$home_dir/$service"
            sudo chown "$target_user:$target_user_group" "$home_dir/$service"
        fi
    done

    # Set up configuration files (if they exist in the source)
    echo "   setting up configuration files..."
    if [[ -d "$source_config" ]]; then
        # Create config directory structure
        sudo -u "$target_user" mkdir -p "$home_dest_config_pyrunner/config"
        sudo -u "$target_user" mkdir -p "$home_dest_config_pyrunner/config/log"
        sudo -u "$target_user" mkdir -p "$home_dest_config_pyrunner/config/simulation"
        
        # Set correct ownership for config directories
        sudo chown -R "$target_user:$target_user_group" "$home_dest_config_pyrunner/config"
    fi

    # Move requirements.txt
    if [[ -f "$source_pypack/requirements.txt" ]]; then
        sudo mv "$source_pypack/requirements.txt" "$home_dest_env/"
        sudo chown "$target_user:$target_user_group" "$home_dest_env/requirements.txt"
    fi

    local dest_fwwriter_wheel=$(find "$home_dest_env" -name "fwwriter-*.whl" -type f 2>/dev/null | head -1)
    local dest_pyrunner_wheel=$(find "$home_dest_env" -name "pyrunner-*.whl" -type f 2>/dev/null | head -1)

    # Run install.sh with the pyrunner wheel as parameter
    if [[ -f "$home_dir/install.sh" ]]; then
        echo "   running install.sh with parameter: $(basename "$dest_pyrunner_wheel"), this will build the PyRunner virtual environment..."
        local initial_dir=$(pwd)
        cd "$home_dir"
        sudo -u "$target_user" bash "$home_dir/install.sh" $(basename "$dest_pyrunner_wheel") &>/dev/null
        if [[ -d "${initial_dir}" ]]; then
            cd $initial_dir
        fi

        # the env was created in home_dest_pyrunner
        sudo chown -R "$target_user:$target_user_group" "$home_dest_pyrunner"
    fi

    # these must exists after install.sh
    if [[ ! -f "$dest_pyrunner_pack_config_HAL" ]] || [[ ! -f "$dest_pyrunner_pack_config_ScanTable" ]] || [[ ! -f "$dest_pyrunner_pack_config_MeasurementUnit" ]]; then
        echo "   ERROR: required PyRunner configuration files not found in $dest_pyrunner_pack_config!"
        echo "   ERROR: installation failed - cannot proceed without configuration files."
        return 1
    fi

    # if any .scaling files in $dest_pyrunner_pack_config, copy them to $home_dest_config_pyrunner
    local dest_pyrunner_config_scaling=$(find "$dest_pyrunner_pack_config" -name "*.scaling" -type f 2>/dev/null | head -1)
    if [[ -f "$dest_pyrunner_config_scaling" ]]; then
        echo "   copying .scaling file(s) from $dest_pyrunner_pack_config to $home_dest_config_pyrunner"
        find "$dest_pyrunner_pack_config" -name "*.scaling" -type f -exec sudo cp {} "$home_dest_config_pyrunner/" \;
    else
        echo "   WARNING: no .scaling file found in $dest_pyrunner_pack_config"
    fi

    echo "   updating configuration files..."

    local inventory_lower=$(echo "$inventory" | tr '[:upper:]' '[:lower:]')

    # Set up configuration files

    local pyrunner_port="8000"
    local logger_port="8113"

    # Get current NodeApplication setting or use default
    local current_nodes=$(get_ini_value "NodeApplication" "HAL" "$dest_pyrunner_pack_config_HAL" 2>/dev/null || echo "")
    echo "    current NodeApplication: $current_nodes"

    # Configure NodeApplication
    str_nodes="Mainboard.cfg;EEFNode.cfg;MC6.cfg"
    
    # Configure Stacker
    if [[ "$(is_in_inventory "$inventory" "Stacker")" = "true" ]]; then
        echo "   configuring Stacker components..."
        str_nodes="${str_nodes};StackerNode.cfg"
        set_ini_value "StackerLeft" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "StackerRight" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling Stacker components..."
        set_ini_value "StackerLeft" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "StackerRight" "0" "HAL" "$dest_pyrunner_pack_config_HAL"

        # Note: Inhibit Stacker node flash
        flash_fw_flags=$(str_to_int "$flash_fw_flags")
        flash_fw_flags="$((flash_fw_flags & -97))"
        flash_fw_flags="$flash_fw_flags"

        force_flash_fw_flags=$(str_to_int "$force_flash_fw_flags")
        force_flash_fw_flags="$((force_flash_fw_flags & -97))"
        force_flash_fw_flags="$force_flash_fw_flags"
    fi

    # Configure Barcode Reader
    if [[ "$(is_in_inventory "$inventory" "BCR")" = "true" ]]; then
        echo "    enabling Barcode Reader upgrade..."
        set_ini_value "BCRUpgrade" "1" "ScanTable" "$dest_pyrunner_pack_config_ScanTable"
    else
        echo "    disabling Barcode Reader upgrade..."
        set_ini_value "BCRUpgrade" "0" "ScanTable" "$dest_pyrunner_pack_config_ScanTable"
    fi

    # Configure TRF Laser
    if [[ "$(is_in_inventory "$inventory" "TRFLaser")" = "true" ]]; then
        echo "    enabling TRF Laser..."
        set_ini_value "TRFLaser" "1" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "TRFLaser_Fan" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling TRF Laser..."
        set_ini_value "TRFLaser" "0" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "TRFLaser_Fan" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
    fi

    # Configure Standard Alpha (Enhanced Alpha)
    if [[ "$(is_in_inventory "$inventory" "StdAlpha")" = "true" ]]; then
        echo "    enabling Standard Alpha..."
        set_ini_value "StdAlpha" "1" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "AlphaLaserStandard_Cooling" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "AlphaLaserStandard_Fan" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling Standard Alpha..."
        set_ini_value "StdAlpha" "0" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "AlphaLaserStandard_Cooling" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "AlphaLaserStandard_Fan" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
    fi

    # Configure HTS Alpha (Ultra Sensitive Alpha)
    if [[ "$(is_in_inventory "$inventory" "HTSAlpha")" = "true" ]]; then
        echo "    enabling HTS Alpha..."
        set_ini_value "HTSAlpha" "1" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "AlphaLaserHTS_Cooling" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
        # Note: AlphaLaserHTS_Fan commented out in original Pascal code
        # set_ini_value "AlphaLaserHTS_Fan" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling HTS Alpha..."
        set_ini_value "HTSAlpha" "0" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "AlphaLaserHTS_Cooling" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
        # Note: AlphaLaserHTS_Fan commented out in original Pascal code
        # set_ini_value "AlphaLaserHTS_Fan" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
    fi

    # Configure US Luminescence (Ultra Sensitive Luminescence)
    # Note: USLum is enabled if either USLum OR HTSAlpha is in inventory
    if [[ "$(is_in_inventory "$inventory" "USLum")" = "true" ]] || [[ "$(is_in_inventory "$inventory" "HTSAlpha")" = "true" ]]; then
        echo "    enabling US Luminescence..."
        set_ini_value "USLum" "1" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "USLumFocusMover" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "USLum_Fan" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling US Luminescence..."
        set_ini_value "USLum" "0" "MeasurementUnit" "$dest_pyrunner_pack_config_MeasurementUnit"
        set_ini_value "USLumFocusMover" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "USLum_Fan" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
    fi

    # Configure FI Bottom
    if [[ "$(is_in_inventory "$inventory" "FIBottom")" = "true" ]]; then
        echo "    enabling FI Bottom..."
        set_ini_value "FI_BOTTOM" "1" "ServicesEnabled" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "LightSwitchEmission" "1" "HAL" "$dest_pyrunner_pack_config_HAL"
    else
        echo "    disabling FI Bottom..."
        set_ini_value "FI_BOTTOM" "0" "ServicesEnabled" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "LightSwitchEmission" "0" "HAL" "$dest_pyrunner_pack_config_HAL"
    fi

    # Set NodeApplication with final configuration
    set_ini_value "NodeApplication" "$str_nodes" "HAL" "$dest_pyrunner_pack_config_HAL"
    
    # Configure Application section
    echo "    configuring application settings..."
    set_ini_value "Version" "$app_version" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "MotFilePath" "$home_dest_mots" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "LogFilePath" "$home_dest_logs" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "GCReportPath" "$home_dest_logs/reports" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "TempPath" "$home_dir/temp" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "GCPort" "$pyrunner_port" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "PyRunnerPort" "$pyrunner_port" "Application" "$dest_pyrunner_pack_config_HAL"

    if [[ "$is_prototyping" == "true" ]]; then
        set_ini_value "AddOn_Script" "$home_dest_hardware_scripts" "Application" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "GC_Task_Export" "$home_dest_config_pyrunner/servicetask.adj" "Application" "$dest_pyrunner_pack_config_HAL"
    else
        set_ini_value "AddOn_Script" "" "Application" "$dest_pyrunner_pack_config_HAL"
        set_ini_value "GC_Task_Export" "" "Application" "$dest_pyrunner_pack_config_HAL"
    fi

    #this line makes no sense for an instrument, but it is in the original code
    set_ini_value "KaleidoIP_Addr" "$(chg_ip_addr "$device_ip" -1)" "Application" "$dest_pyrunner_pack_config_HAL"
    
    set_ini_value "DjangoIP_Addr" "$device_ip" "Application" "$dest_pyrunner_pack_config_HAL"
    set_ini_value "LoggerPort" "$logger_port" "Application" "$dest_pyrunner_pack_config_HAL"  
    
    # Configure Instrument section
    echo "    configuring instrument settings..."
    set_ini_value "SerNo" "0815" "Instrument" "$dest_pyrunner_pack_config_HAL" 
    set_ini_value "Inventory" "$inventory_lower" "Instrument" "$dest_pyrunner_pack_config_HAL"

    # Update logger configuration files
    if [[ -f "$dest_pyrunner_pack_config_logger_server" ]]; then
        echo "    updating logger server configuration..."
        # Replace filename path in logger_server.yaml
        sed -i "s|filename: |filename: $home_dir/logs/|g" "$dest_pyrunner_pack_config_logger_server" 2>/dev/null || true
    fi

    # update logger configuration level
    if [[ -f "$dest_pyrunner_pack_config_logger" ]]; then
        if [[ "$is_develop" = "true" ]]; then
            echo "    updating logger configuration to DEBUG level..."
            sed -i 's/level: INFO/level: DEBUG/g' "$dest_pyrunner_pack_config_logger" 2>/dev/null || true
        else
            echo "    updating logger configuration to INFO level..."
            sed -i 's/level: DEBUG/level: INFO/g' "$dest_pyrunner_pack_config_logger" 2>/dev/null || true
        fi
    fi

    # Update webinterface [django] configuration
    if [[ -f "$dest_pyrunner_pack_config_django" ]]; then
        echo "    updating webinterface [django] configuration..."
    else
        echo "    WARNING: webinterface [django] configuration file not found at $dest_pyrunner_pack_config_django, creating it"
        touch "$dest_pyrunner_pack_config_django"
    fi
    set_ini_value "LogFilePath" "$home_dest_logs" "Directories" "$dest_pyrunner_pack_config_django"

    local dest_pyrunner_config_measurement_unit_dir=""
    if [[ "$is_simulation" == "true" ]]; then
        dest_pyrunner_config_measurement_unit_dir="$dest_pyrunner_pack_config"
    else
        dest_pyrunner_config_measurement_unit_dir="$home_dest_config_pyrunner"
    fi
    local dest_pyrunner_config_measurement_unit="$dest_pyrunner_config_measurement_unit_dir/MeasurementUnit.adj"
    if [[ -f "$dest_pyrunner_config_measurement_unit" ]]; then
        echo "    updating MeasurementUnit.adj configuration..."

	    local strParam=$(get_ini_value 'HighVoltageSetting_PMT_HTSAlpha' 'GC_Params' "$dest_pyrunner_config_measurement_unit" "None")
        if [[ "$strParam" == "None" ]]; then

            strParam=$(get_ini_value 'NormFactor_US_PMT' 'GC_Params'  "$dest_pyrunner_config_measurement_unit" "1.0")
            set_ini_value 'NormFactor_HTSAlpha_PMT' 'GC_Params' "$strParam" "$dest_pyrunner_config_measurement_unit"

            strParam=$(get_ini_value 'HighVoltageSetting_PMT_USLUM' 'GC_Params' "$dest_pyrunner_config_measurement_unit" "0.68")
            if [[ $(is_str_a_number "$strParam") == "true" ]]; then
                strParam=$(str_multiply "$strParam" "1.2")
                set_ini_value 'HighVoltageSetting_PMT_HTSAlpha' 'GC_Params' "$strParam" "$dest_pyrunner_config_measurement_unit"
            fi

            strParam=$(get_ini_value 'DiscriminatorLevel_PMT_USLUM' 'GC_Params' "$dest_pyrunner_config_measurement_unit" "0.38")
            set_ini_value 'DiscriminatorLevel_PMT_HTSAlpha' 'GC_Params' "$strParam" "$dest_pyrunner_config_measurement_unit"

            strParam='0.000000008'
            set_ini_value 'Pulse_pair_res_PMT_HTSAlpha_s' 'GC_Params' "$strParam" "$dest_pyrunner_config_measurement_unit"
        fi
    else
        echo "    WARNING: MeasurementUnit.adj file not found at $dest_pyrunner_config_measurement_unit"
    fi
    set_ini_value "AdjFilePath" "$dest_pyrunner_config_measurement_unit_dir" "Application" "$dest_pyrunner_pack_config_HAL"
  
    if [[ -f "$current_pyrunner_pack_config_MeasurementUnit" ]]; then
        local strParam=$(get_ini_value 'GC_Params' 'HTSAlphaLaserPower' "$current_pyrunner_pack_config_MeasurementUnit" "0.75")
        set_ini_value 'GC_Params' 'HTSAlphaLaserPower' "0.75" "$dest_pyrunner_pack_config_MeasurementUnit" 
    fi

    # Update TemperatureServices configuration
    if [[ -f "$dest_pyrunner_pack_config_TemperatureServices" ]]; then
        echo "    updating TemperatureServices configuration..."
        set_ini_value "sensor_scaling_path"  "$home_dest_config_pyrunner/temperature.scaling" "Measurement_Chamber" "$dest_pyrunner_pack_config_TemperatureServices"
        set_ini_value "setpoint_scaling_path" "$home_dest_config_pyrunner/setpoint.scaling" "Measurement_Chamber" "$dest_pyrunner_pack_config_TemperatureServices"
    else
        echo "    WARNING: TemperatureServices configuration file not found at $dest_pyrunner_pack_config_TemperatureServices, skipping temperature service update"
    fi

    # configure django.service
    if [[ -f "$home_dir/django.service" ]]; then
        echo "    updating webinterface [django] service configuration..."
        set_ini_value "ExecStart" "$home_dir/pyrunner/bin/python3 manage.py runserver 0.0.0.0:$pyrunner_port" "Service" "$home_dir/django.service"
        set_ini_value "WorkingDirectory" "$dest_pyrunner_pack_dir/webinterface" "Service" "$home_dir/django.service"
    fi

    # configure pyrunner.service
    if [[ -f "$home_dir/pyrunner.service" ]]; then
        echo "    updating pyrunner.service configuration..."
        set_ini_value "User" "$target_user" "Service"  "$home_dir/pyrunner.service"
        set_ini_value "WorkingDirectory" "$dest_pyrunner_pack_dir" "Service" "$home_dir/pyrunner.service"
        
        local cmd=("$home_dir/pyrunner/bin/python3" "pyrunner.py")
        if [[ "$is_simulation" == "true" ]]; then
            cmd+=("--simulation")
        fi
        cmd+=("--kaleido")

        # Nov 2025: Changes to keep groundcontrol active 
        cmd+=("--groundcontrol")

        cmd+=("127.0.0.1:$pyrunner_port")
        local set_value="${cmd[*]}"
        set_ini_value "ExecStart" "$set_value" "Service" "$home_dir/pyrunner.service"
    fi

    # configure chownwdog.service
    if [[ -f "$home_dir/chownwdog.service" ]]; then
        echo "    updating chownwdog.service configuration..."
        set_ini_value "User" "root" "Service"  "$home_dir/chownwdog.service"
        set_ini_value "ExecStart" "/bin/chown $target_user /dev/watchdog" "Service" "$home_dir/chownwdog.service"
    fi

    if [[ -f "$home_dir/chkwho.sh" ]]; then
        echo "   copying chkwho.sh script to /etc/profile.d..."
        sudo cp -f "$home_dir/chkwho.sh" "/etc/profile.d/chkwho.sh"
        sudo chmod 744 "/etc/profile.d/chkwho.sh"
        sudo chown "root:root" "/etc/profile.d/chkwho.sh"
    fi

    # set correct ownership for config files
    sudo chown -R "$target_user:$target_user_group" "$dest_pyrunner_pack_config"
    sudo chown -R "$target_user:$target_user_group" "$home_dest_config_pyrunner"
    sudo chown "$target_user:$target_user_group" "$dest_pyrunner_pack_config_django"
    
    echo "   installing services with systemd..."
    # instead of Run installsystemd.sh
    if [[ -f "$home_dir/pyrunner.service" ]]; then
        echo "    pyrunner.service..."
        mv -f "$home_dir/pyrunner.service" "$pyrunner_systemd"
        #sudo chown root:root "$pyrunner_systemd"
        systemctl daemon-reload
        systemctl enable pyrunner.service > /dev/null 2>&1
    fi
    if [[ -f "$home_dir/django.service" ]]; then
        echo "    django.service..."
        mv -f "$home_dir/django.service" "$django_systemd"
        #sudo chown root:root "$django_systemd"
        systemctl daemon-reload
        systemctl enable django.service > /dev/null 2>&1
    fi
    if [[ -f "$home_dir/chownwdog.service" ]]; then
        echo "    chownwdog.service..."
        mv -f "$home_dir/chownwdog.service" "$chownwdog_systemd"
        #sudo chown root:root "$chownwdog_systemd"
        systemctl daemon-reload
        systemctl enable chownwdog.service > /dev/null 2>&1
    fi

    # Flash Firmware
    if [[ ! "$flash_fw_flags" == "0" ]]; then
        echo "   uploading Firmware..."
        local index=0
        if ! flash_fw "$flash_fw_flags" "false" "$home_dir"; then
            index=$?
            echo "   WARNING: Flash Firmware causes problems: $index"
        fi
    fi

    # Forced upload Firmware (if needed)
    if [[ ! "$force_flash_fw_flags" == "0" ]]; then
        echo "   force uploading Firmware..."
        local force_index=0
        if ! flash_fw "$force_flash_fw_flags" "true" "$home_dir"; then
            force_index=$?
            echo "   WARNING: Forced Flash Firmware causes problems: $force_index"
        fi
    fi
}

remove_pyrunner() {
    # Initialize default values
    local target_user=""
    local pyrunner_systemd=""
    local django_systemd=""
    local chownwdog_systemd=""
    
    # Parse named parameters
    for param in "$@"; do
        case "$param" in
            target_user=*)
                target_user="${param#target_user=}"
                ;;
            pyrunner_systemd=*)
                pyrunner_systemd="${param#pyrunner_systemd=}"
                ;;
            django_systemd=*)
                django_systemd="${param#django_systemd=}"
                ;;
            chownwdog_systemd=*)
                chownwdog_systemd="${param#chownwdog_systemd=}"
                ;;
            *)
                echo "   WARNING: unknown parameter: $param (ignoring)"
                ;;
        esac
    done
    
    # Validate required parameters
    if [[ -z "$target_user" ]]; then
        echo "   ERROR: targetUser parameter is required!"
        return 1
    fi
    if [[ -z "$pyrunner_systemd" ]]; then
        echo "   ERROR: pyrunnerSystemdPath parameter is required!"
        return 1
    fi
    if [[ -z "$django_systemd" ]]; then
        echo "   ERROR: djangoSystemdPath parameter is required!"
        return 1
    fi
    if [[ -z "$chownwdog_systemd" ]]; then
        echo "   ERROR: chownwdogSystemdPath parameter is required!"
        return 1
    fi
    
    local home_dir="/home/${target_user}"
    
    local pyrinstall="${home_dir}/.PyRInstall"
    local env="${pyrinstall}/InstEnv"
    local mots="${pyrinstall}/mots"
    local protocols="${pyrinstall}/PresetProtocols"
    local temp="${home_dir}/temp"
    local logs="${home_dir}/logs"
    local config="${home_dir}/.config/pyrunner"
    local pyrunner="${home_dir}/pyrunner"
    local hardware_scripts="${home_dir}/hardware_scripts"
    local dot_local_virtualenv_dir="${home_dir}/.local/share/virtualenv/"

    # Remove directories
    directories_to_remove=(
        "$temp"
        "$logs"
        "$config"
        "$pyrinstall"
        "$pyrunner"
        "$hardware_scripts"
        "$dot_local_virtualenv_dir"
    )

    # Remove files
    files_to_remove=(
        "$home_dir/chkwho.sh"
        "$home_dir/FWUpdate.sh"
        "$home_dir/install.sh"
        "$home_dir/installsystemd.sh"
        "$home_dir/RestartPyRunner.sh"
        "$home_dir/StartPyRunner.sh"
        "$home_dir/StopPyRunner.sh"
        "$home_dir/TermPy.sh"
        "$home_dir/uninstall.sh"
        "$home_dir/use_dhcp.sh"
        "$home_dir/chownwdog.service"
        "$home_dir/django.service"
        "$home_dir/pyrunner.service"
        "$home_dir/merge_fi_bottom_to_gc.sh"
    )

    # Remove directories
    for directory in "${directories_to_remove[@]}"; do
        if [[ -d "$directory" ]]; then
            echo "  removing pyrunner directory: $directory..."
            sudo rm -rf "$directory"
        fi
    done

    # Remove files
    for file in "${files_to_remove[@]}"; do
        if [[ -f "$file" ]]; then
            echo "  removing pyrunner file: $file..."
            sudo rm -f "$file"
        fi
    done

  if [[ -f "${pyrunner_systemd}" ]]; then
      echo "  pyrunner systemd service file ${pyrunner_systemd} exists, delete it..."
      rm -f "${pyrunner_systemd}"
      systemctl daemon-reload
  fi

  if [[ -f "${django_systemd}" ]]; then
      echo "  pyrunner django systemd service file ${django_systemd} exists, delete it..."
      rm -f "${django_systemd}"
      systemctl daemon-reload
  fi

  if [[ -f "${chownwdog_systemd}" ]]; then
      echo "  chownwdog systemd service file ${chownwdog_systemd} exists, delete it..."
      rm -f "${chownwdog_systemd}"
      systemctl daemon-reload
  fi

}

# Flash firmware for PyRunner

# Firmware flash control constants (bitwise flags)
readonly FLASH_None=0          # | . | . | . | . | . | . | . | . | . |
readonly EEF_Node=1            # | . | . | . | . | . | . | . | . | 1 |
readonly EEF_BL=2              # | . | . | . | . | . | . | . | 1 | . |
readonly EEF_FPGA=4            # | . | . | . | . | . | . | 1 | . | . |
readonly MC6_Node=8            # | . | . | . | . | . | 1 | . | . | . |
readonly MC6_BL=16             # | . | . | . | . | 1 | . | . | . | . |
readonly MC6_STK_Node=32       # | . | . | . | 1 | . | . | . | . | . |
readonly MC6_STK_BL=64         # | . | . | 1 | . | . | . | . | . | . |
readonly FMB_Node=128          # | . | 1 | . | . | . | . | . | . | . |
readonly FMB_BL=256            # | 1 | . | . | . | . | . | . | . | . |
readonly FLASH_All=511         # | 1 | 1 | 1 | 1 | 1 | 1 | 1 | 1 | 1 |

# Normal FW call: EEF_Node + MC6_Node + MC6_STK_Node + FMB_Node = "169"
#                     1    +     8     +      32      +    128   = "169"

# Function to flash firmware based on control flags
# Parameters:
#   control_value - integer bitwise control value
#   force_flag - boolean (true/false) to force flash operation
#   home_dir - path to home directory where FWUpdate.sh resides
# Returns: accumulated error count from all flash operations
flash_fw() {
    local control_value="$1"
    local force_flag="$2"
    local home_dir="$3"
    
    # Validate parameters
    if [[ -z "$control_value" ]]; then
        echo "   ERROR: flash_fw requires control_value parameter"
        return 1
    fi
    if [[ -z "$force_flag" ]]; then
        echo "   ERROR: flash_fw requires force_flag parameter"
        return 1
    fi
    if [[ "$force_flag" != "true" && "$force_flag" != "false" ]]; then
        echo "   ERROR: flash_fw force_flag must be 'true' or 'false', got: '$force_flag'"
        return 1
    fi
    if [[ -z "$home_dir" ]]; then
        echo "   ERROR: flash_fw requires home_dir parameter"
        return 1
    fi
    
    local result=0
    local index=0
    local str_param=""
    local str_shell_cmd="$home_dir/FWUpdate.sh"
    local str_force=""
    
    # Set force flag parameter
    if [[ "$force_flag" == "true" ]]; then
        str_force="--force"
    else
        str_force=""
    fi
    
    # Check if FWUpdate.sh exists
    if [[ ! -f "$str_shell_cmd" ]]; then
        echo "   ERROR: FWUpdate.sh not found at $str_shell_cmd"
        return 1
    fi
    
    # Check if all required firmware files exist
    local mots_dir="$home_dir/.PyRInstall/mots"
    local firmware_files=(
        "$mots_dir/EEF_BDH_Node.mot"
        "$mots_dir/Bootloader.mot" 
        "$mots_dir/EEF_FPGA.mot"
        "$mots_dir/MC6_Node.mot"
        "$mots_dir/FMB_Node.mot"
    )
    
    local missing_files=()
    for firmware_file in "${firmware_files[@]}"; do
        if [[ ! -f "$firmware_file" ]]; then
            missing_files+=("$firmware_file")
        fi
    done
    
    if [[ ${#missing_files[@]} -gt 0 ]]; then
        echo "   WARNING: Some firmware files are missing:"
        for missing_file in "${missing_files[@]}"; do
            echo "   WARNING: Missing firmware file: $missing_file"
        done
        echo "   WARNING: Firmware flashing may fail for missing components"
    fi
    
    # Flash EEF_Node if flag is set
    if (( (EEF_Node & control_value) == EEF_Node )); then
        echo "    Flash EEF_Node"
        str_param="EEFNode $mots_dir/EEF_BDH_Node.mot 1 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash EEF Bootloader if flag is set
    if (( (EEF_BL & control_value) == EEF_BL )); then
        echo "    Flash EEF Bootloader"
        str_param="EEFNode $mots_dir/Bootloader.mot 0 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash EEF FPGA if flag is set
    if (( (EEF_FPGA & control_value) == EEF_FPGA )); then
        echo "    Flash EEF FPGA"
        str_param="EEFNode $mots_dir/EEF_FPGA.mot 2 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash MC6_Node if flag is set
    if (( (MC6_Node & control_value) == MC6_Node )); then
        echo "    Flash MC6_Node"
        str_param="MC6 $mots_dir/MC6_Node.mot 1 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash MC6 Bootloader if flag is set
    if (( (MC6_BL & control_value) == MC6_BL )); then
        echo "    Flash MC6_Bootloader"
        str_param="MC6 $mots_dir/Bootloader.mot 0 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash MC6_STK_Node if flag is set
    if (( (MC6_STK_Node & control_value) == MC6_STK_Node )); then
        echo "    Flash MC6_STK_Node"
        str_param="StackerNode $mots_dir/MC6_Node.mot 1 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash MC6_STK Bootloader if flag is set
    if (( (MC6_STK_BL & control_value) == MC6_STK_BL )); then
        echo "    Flash MC6_STK_Bootloader"
        str_param="StackerNode $mots_dir/Bootloader.mot 0 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash FMB_Node if flag is set
    if (( (FMB_Node & control_value) == FMB_Node )); then
        echo "    Flash FMB_Node"
        str_param="Mainboard $mots_dir/FMB_Node.mot 1 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    # Flash FMB Bootloader if flag is set
    if (( (FMB_BL & control_value) == FMB_BL )); then
        echo "    Flash FMB_Bootloader"
        str_param="Mainboard $mots_dir/Bootloader.mot 0 $str_force"
        str_param="${str_param%% }"  # trim trailing spaces
        index=0
        if ! "$str_shell_cmd" $str_param > /dev/null; then
            index=$?
            echo "   WARNING: $str_shell_cmd $str_param causes a problem: $index"
        fi
        result=$((result + index))
    fi
    
    return $result
}
