stop_services() {
  local what=${1:-"yes-all"}

  if [[ "$what" != "yes" ]] && [[ "$what" != "yes-all" ]] && [[ "$what" != "yes-all-except-control" ]] && [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
    return 0
  fi

  if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    stop_and_disable_nginx_service
  fi
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    if [[ "$what" == "yes-all-except-control" ]] || [[ "$what" == "yes-all-except-control-pyrunner" ]]; then
      disable_nginx_bodhi_proxy yes no yes
    else
      disable_nginx_bodhi_proxy yes yes yes
    fi
  fi
  stop_bodhi_apps_service
  
  if [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
    stop_pyrunner_services
  fi

  if [[ "$what" != "yes-all-except-control" ]] && [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
    stop_bodhi_control_service
  fi
}

start_services() {
  local what=${1:-"yes-all"}

  if [[ "$what" != "yes" ]] && [[ "$what" != "yes-all" ]] && [[ "$what" != "yes-all-except-control" ]] && [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
    return 0
  fi

  if [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
    start_pyrunner_services
    #give the services time to start 
    sleep 5
  fi

  if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    stop_and_disable_nginx_service
    start_bodhi_apps_service

    #give the services time to start 
    sleep 15
  fi
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    if [[ "$what" != "yes-all-except-control" ]] && [[ "$what" != "yes-all-except-control-pyrunner" ]]; then
      start_bodhi_control_service
    fi
    start_bodhi_apps_service

    #give the services time to start 
    sleep 15

    if [[ "$what" == "yes-all-except-control" ]] || [[ "$what" == "yes-all-except-control-pyrunner" ]]; then
      enable_nginx_bodhi_proxy yes no no
    else
      enable_nginx_bodhi_proxy yes yes no
    fi
    enable_and_start_nginx_service
  fi
}

reinitialize_instrument()
{
  # rehome the hardware and restart the software
  echo "  reinitializing instrument: will rehome the hardware and restart the software..."
  
  # wait a bit before shutting down the services
  # to allow the potential caller to graciously complete before being killed
  sleep 5

  stop_services yes-all
  restart_postgres_service
  start_services yes-all
}
 
stop_apps_processes() {
  if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    stop_nodejs_processes
  fi
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    stop_dotnet_apps_processes
  fi
  stop_pyreader_processes
}

start_app_processes() {
  start_pyreader_process
  if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    start_nodejs_processes
  fi
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    start_dotnet_apps_processes
  fi
}

stop_control_processes() {
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    stop_dotnet_control_processes
  fi
}

# this function is the one that is called at system boot by bodhi-control service
start_control_processes() {
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    start_dotnet_control_processes

    # Check if within 3 minutes after system boot
    local uptime_seconds=$(awk '{print int($1)}' /proc/uptime)
    if [[ $uptime_seconds -le 180 ]]; then
      # Wait an additional 60 seconds to ensure system stability
      # and that the other services are fully started
      sleep 60

      echo "  *** System booted $uptime_seconds seconds ago. Making sure nginx is enabled to serve the bodhi-control app..."
      enable_nginx_bodhi_proxy yes yes no
      enable_and_start_nginx_service
    fi
  fi
}

start_nodejs_processes() {
  if [[ -d /bodhi/run/current/apps/bodhi-api ]] && [[ -d /bodhi/run/current/apps/bodhi-ui ]]; then
    echo "  Starting bodhi-api and bodhi-ui apps..."
    initial_dir=$(pwd)
    cd /bodhi/run/current/apps/bodhi-api
    nohup node start.js > /dev/null 2>&1 &
    cd $initial_dir
  else
    if [[ ! -d /bodhi/run/current/apps/bodhi-dotnet ]]; then
      echo "  ERROR: missing bodhi-api at /bodhi/run/current/apps/bodhi-api"
    fi
    if [[ ! -d /bodhi/run/current/apps/bodhi-ui ]]; then
      echo "  ERROR: missing bodhi-ui at /bodhi/run/current/apps/bodhi-ui"
    fi
  fi
}

stop_nodejs_processes() {
  echo "Stopping bodhi-nodejs apps..."
  for PID in $(ps ax -o pid,stat,command | grep -v grep | grep node | grep start.js | grep ' Ss ' | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      echo "The node start.js process with PID $PID is running. Attempting to stop..."
      kill $PID
    fi
  done

  # loop and check every second if the process is still alive
  for i in {1..45}
  do
    local done=true
    for PID in $(ps ax | grep -v grep | grep node | grep start.js | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        done=false
        break
      fi
    done
    
    if [[ "${done}" == "true" ]]; then
      echo "All processes were killed"
      break
    else
      sleep 1
    fi
  done

  for PID in $(ps ax | grep -v grep | grep node | grep start.js | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      echo "The node start.js process with PID $PID is still running. Attempting to SIGKILL kill..."
      kill -9 $PID
    fi
  done
}

pyreader_service_run_args='./calculation/manage.py runserver 127.0.0.1:8080'
start_pyreader_process() {
  if [[ -d /bodhi/run/current/apps/swp-PyReader ]]; then
    echo "Starting bodhi-pyreader app..."
    initial_dir=$(pwd)
    cd /bodhi/run/current/apps/swp-PyReader
    nohup ./VirtualEnv/bin/python $pyreader_service_run_args > /dev/null 2>&1 &
    cd $initial_dir
  else
    echo "ERROR: missing pyreader at /bodhi/run/current/apps/swp-PyReader"
  fi
}
stop_pyreader_processes() {
  echo "Stopping bodhi-pyreader app..."
  for PID in $(ps ax -o pid,stat,command | grep -v grep | grep python | grep "$pyreader_service_run_args" | grep ' S[s|l] ' | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      echo "The python $pyreader_service_run_args process with PID $PID is running. Attempting to stop..."
      kill $PID
    fi
  done

  # loop and check every second if the process is still alive
  for i in {1..45}
  do
    local done=true
    for PID in $(ps ax | grep -v grep | grep python | grep "$pyreader_service_run_args" | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        done=false
        break
      fi
    done
   
    if [[ "${done}" == "true" ]]; then
      echo "All processes were killed"
      break
    else
      sleep 1
    fi
  done

  for PID in $(ps ax | grep -v grep | grep python | grep "$pyreader_service_run_args" | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      echo "The python $pyreader_service_run_args process with PID $PID is still running. Attempting to SIGKILL kill..."
      kill -9 $PID
    fi
  done
}

start_dotnet_apps_processes() {
  if [[ -d "/bodhi/run/current/apps/bodhi-dotnet" ]] && [[ -d /bodhi/run/current/apps/bodhi-ui ]]; then
    initial_dir=$(pwd)
    echo "Starting all bodhi-dotnet apps..."
    cd "/bodhi/run/current/apps/bodhi-dotnet"
    cd ./auth && ( nohup ./MMPR.SimpleIdentityProviderHost > /dev/null 2>&1 & ) && cd ..
    cd ./api && ( nohup ./MMPR.API --urls http://0.0.0.0:5041 > /dev/null 2>&1 & ) && cd ..
    cd ./ws && ( nohup ./MMPR.WebSocketServer > /dev/null 2>&1 & ) && cd ..
    cd ./ui && ( nohup ./MMPR.WebUiServer > /dev/null 2>&1 & ) && cd ..
    cd $initial_dir
  else
    echo "ERROR: missing bodhi-dotnet apps at /bodhi/run/current/apps/bodhi-dotnet"
  fi
}

stop_dotnet_apps_processes() {
  echo "Stopping all bodhi-dotnet apps..."
  local apps=(
    "MMPR.SimpleIdentityProviderHost"
    "MMPR.API"
    "MMPR.WebSocketServer"
    "MMPR.WebUiServer"
  )

  for app in "${apps[@]}"; do
    for PID in $(ps ax -o pid,stat,command | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        echo "A dotnet ${app} process with PID $PID is running. Attempting to stop..."
        kill $PID
      fi
    done
  done

  # loop and check every second if the process is still alive
  for i in {1..45
  do
    local done=true
    for app in "${apps[@]}"; do
      for PID in $(ps ax -o pid,stat,command | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
        if ps -p $PID > /dev/null 2>&1; then
          done=false
          break
        fi
      done
    done
    
    if [[ "${done}" == "true" ]]; then
      echo "All processes were killed"
      break
    else
      sleep 1
    fi
  done

  for app in "${apps[@]}"; do
    for PID in $(ps ax | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        echo "A dotnet ${app} process with PID $PID is still running. Attempting to SIGKILL kill..."
        kill -9 $PID
      fi
    done
  done
}

start_dotnet_control_processes() {
  if [[ -d "/bodhi/run/current/control" ]]; then
    initial_dir=$(pwd)
    echo "  Starting all bodhi-control apps..."
    cd "/bodhi/run/current/control/bodhi-dotnet"
    cd ./control && ( nohup ./MMPR.ControlService > /dev/null 2>&1 & ) && cd ..
    cd $initial_dir
  else
    echo "ERROR: missing bodhi-control apps at /bodhi/run/current/control"
  fi
}

stop_dotnet_control_processes() {
  echo "Stopping all bodhi-control apps..."
  local apps=(
    "MMPR.ControlService"
  )

  for app in "${apps[@]}"; do
    for PID in $(ps ax -o pid,stat,command | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        echo "A dotnet ${app} process with PID $PID is running. Attempting to stop..."
        kill $PID
      fi
    done
  done

  # loop and check every second if the process is still alive
  for i in {1..45
  do
    local done=true
    for app in "${apps[@]}"; do
      for PID in $(ps ax | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
        if ps -p $PID > /dev/null 2>&1; then
          done=false
          break
        fi
      done
    done
   
    if [[ "${done}" == "true" ]]; then
      echo "All processes were killed"
      break
    else
      sleep 1
    fi
  done

  for app in "${apps[@]}"; do
    for PID in $(ps ax | grep -v grep | grep ${app} | grep ' Sl ' | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        echo "A dotnet ${app} process with PID $PID is still running. Attempting to SIGKILL kill..."
        kill -9 $PID
      fi
    done
  done
}

stop_and_disable_nginx_service() {
  local stop=${1:-"yes"}
  local disable=${2:-"yes"}
  if [[ -f "${nginx_conf}/nginx.conf" ]]; then
    if command -v nginx >/dev/null 2>&1; then
      if [[ "$stop" == "yes" ]]; then
        if systemctl is-active --quiet nginx.service; then
          echo "  nginx is running, stopping now..."
          systemctl stop nginx.service > /dev/null 2>&1
        fi
      fi
      if [[ "$disable" == "yes" ]]; then
        if systemctl is-enabled --quiet nginx.service; then
          echo "  nginx is enabled, disabling now..."
          systemctl disable nginx.service > /dev/null 2>&1
        fi
      fi
    fi
  fi
}

enable_and_start_nginx_service() {
  local enable=${1:-"yes"}
  local start=${2:-"yes"}
  local reload=${3:-"yes"}

  if [[ -f "${nginx_conf}/nginx.conf" ]]; then
    if command -v nginx >/dev/null 2>&1; then
      local started=false
    
      if [[ "$enable" == "yes" ]]; then
        if ! systemctl is-enabled --quiet nginx.service; then
            echo "  nginx is not enabled, enabling now..."
            systemctl enable nginx.service > /dev/null 2>&1
        fi
      fi
    
      if [[ "$start" == "yes" ]]; then
        if ! systemctl is-active --quiet nginx.service; then
          echo "  nginx is not running, starting now..."
          systemctl start nginx.service > /dev/null 2>&1
          started=true
        fi
      fi

      if [[ "$reload" == "yes" ]]; then
        if [[ "$started" == "false" ]]; then
          echo "  reloading the nginx configuration on enable and start nginx..."
          nginx -s reload  > /dev/null 2>&1
        fi
      fi

    else
        echo "  WARNING: nginx is not installed!"
    fi
  else
    echo "  WARNING: nginx config file ${nginx_conf}/nginx.conf not present!"
  fi
}

restart_postgres_service() {
  local enable=${1:-"yes"}
  local start=${2:-"yes"}
  if command -v psql >/dev/null 2>&1; then
    if [[ "$enable" == "yes" ]]; then
      if ! systemctl is-enabled --quiet postgresql.service; then
          echo "  postgresql is not enabled, enabling now..."
          systemctl enable postgresql.service > /dev/null 2>&1
      fi
    fi
    if [[ "$start" == "yes" ]]; then
      if systemctl is-active --quiet postgresql.service; then
        echo "  postgresql is running, restarting now..."
        systemctl restart postgresql.service > /dev/null 2>&1
      else
        echo "  postgresql is not running, starting now..."
        systemctl start postgresql.service > /dev/null 2>&1
      fi
    fi
  else
    echo "  WARNING: postgresql is not installed!"
  fi
}

stop_bodhi_apps_service() {
  if [[ -f ${bodhi_apps_initd} ]]; then
      if systemctl is-active --quiet bodhi-apps.service; then
        echo "  bodhi-apps service is running, stopping now..."
        systemctl stop bodhi-apps.service > /dev/null 2>&1
      fi
  fi
}

start_bodhi_apps_service() {
  if [[ -f ${bodhi_apps_initd} ]]; then
      if ! systemctl is-active --quiet bodhi-apps.service; then
        echo "  bodhi-apps service is not running, starting now..."
        systemctl start bodhi-apps.service > /dev/null 2>&1
      fi
  fi
}

stop_bodhi_control_service() {
  if [[ -f ${bodhi_control_initd} ]]; then
      if systemctl is-active --quiet bodhi-control.service; then
        echo "  bodhi-control service is running, stopping now..."
        systemctl stop bodhi-control.service > /dev/null 2>&1
      fi
  fi
}

start_bodhi_control_service() {
  if [[ -f ${bodhi_control_initd} ]]; then
      if ! systemctl is-active --quiet bodhi-control.service; then
        echo "  bodhi-control service is not running, starting now..."
        systemctl start bodhi-control.service > /dev/null 2>&1
      fi
  fi
}

create_bodhi_apps_service() {
  write_bodhi_apps_services=false
  if [[ -f ${bodhi_apps_initd} ]]; then
    if [[ "$CREATE_SERVICES" == "overwrite" ]]; then
      echo "  bodhi-apps service exists, overwrite it"
      write_bodhi_apps_services=true
    else
      echo "  bodhi-apps service exists, do not overwrite it"
      write_bodhi_apps_services=false
    fi
  else
    echo "  bodhi-apps service is not present, create it"
    write_bodhi_apps_services=true
  fi

  if [[ "$write_bodhi_apps_services" == "true" ]]; then
    echo '#!/bin/sh
#BODHI_APPS_INITD
### BEGIN INIT INFO
# Provides:          bodhi apps
# Required-Start:    $remote_fs $syslog
# Required-Stop:     $remote_fs $syslog
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Starts /Stops bodhi apps
# Description:       Start / stop bodhi apps at boot / shutdown.
### END INIT INFO

case "$1" in
    start)
        echo "Starting bodhi apps"
        cd "BODHI_SCRIPTS_FOLDER"
        BODHI_APPS_SH --framework-language FRAMEWORK_LANGUAGE --start-apps-processes yes
        ;;
    stop)
        echo "Stopping bodhi apps"
        cd "BODHI_SCRIPTS_FOLDER"
        BODHI_APPS_SH --framework-language FRAMEWORK_LANGUAGE --stop-apps-processes yes
        ;;
    *)
        echo "Usage: BODHI_APPS_INITD {start|stop}"
        exit 1
        ;;
esac
exit 0
  ' > ${bodhi_apps_initd}

    local bodhi_scripts_folder="${bodhi_folder_run}/current/scripts"
    local bodhi_apps_sh="./bodhi-apps.sh"

    sed -i "s|BODHI_SCRIPTS_FOLDER|${bodhi_scripts_folder}|g" "${bodhi_apps_initd}"
    sed -i "s|BODHI_APPS_SH|${bodhi_apps_sh}|g" "${bodhi_apps_initd}"

    sed -i "s|FRAMEWORK_LANGUAGE|${FRAMEWORK_LANGUAGE}|g" "${bodhi_apps_initd}"
    sed -i "s|BODHI_APPS_INITD|${bodhi_apps_initd}|g" "${bodhi_apps_initd}"

    chmod +x ${bodhi_apps_initd}
    update-rc.d bodhi-apps defaults
    systemctl daemon-reload > /dev/null 2>&1
  fi
}

create_bodhi_control_service() {
  write_bodhi_control_services=false
  if [[ -f ${bodhi_control_initd} ]]; then
    if [[ "$CREATE_SERVICES" == "overwrite" ]]; then
      echo "  bodhi-control service exists, overwrite it"
      write_bodhi_control_services=true
    else
      echo "  bodhi-control service exists, do not overwrite it"
      write_bodhi_control_services=false
    fi
  else
    echo "  bodhi-control service is not present, create it"
    write_bodhi_control_services=true
  fi

  if [[ "$write_bodhi_control_services" == "true" ]]; then
    echo '#!/bin/sh
#BODHI_CONTROL_INITD
### BEGIN INIT INFO
# Provides:          bodhi control
# Required-Start:    $remote_fs $syslog
# Required-Stop:     $remote_fs $syslog
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Starts /Stops bodhi control
# Description:       Start / stop bodhi control at boot / shutdown.
### END INIT INFO

case "$1" in
    start)
        echo "Starting bodhi control"
        cd "BODHI_SCRIPTS_FOLDER"
        BODHI_APPS_SH --framework-language FRAMEWORK_LANGUAGE --start-control-processes yes
        ;;
    stop)
        echo "Stopping bodhi control"
        cd "BODHI_SCRIPTS_FOLDER"
        BODHI_APPS_SH --framework-language FRAMEWORK_LANGUAGE --stop-control-processes yes
        ;;
    *)
        echo "Usage: BODHI_CONTROL_INITD {start|stop}"
        exit 1
        ;;
esac
exit 0
  ' > ${bodhi_control_initd}

    local bodhi_scripts_folder="${bodhi_folder_run}/current/scripts"
    local bodhi_apps_sh="./bodhi-apps.sh"

    sed -i "s|BODHI_SCRIPTS_FOLDER|${bodhi_scripts_folder}|g" "${bodhi_control_initd}"
    sed -i "s|BODHI_APPS_SH|${bodhi_apps_sh}|g" "${bodhi_control_initd}"

    sed -i "s|FRAMEWORK_LANGUAGE|${FRAMEWORK_LANGUAGE}|g" "${bodhi_control_initd}"
    sed -i "s|BODHI_CONTROL_INITD|${bodhi_control_initd}|g" "${bodhi_control_initd}"

    chmod +x ${bodhi_control_initd}
    update-rc.d bodhi-control defaults
    systemctl daemon-reload > /dev/null 2>&1
  fi
}

delete_bodhi_control_service() {
  if [[ -f "${bodhi_control_initd}" ]]; then
    rm -f "${bodhi_control_initd}"
    systemctl daemon-reload > /dev/null 2>&1
  fi
}

create_nginx_services() {
  write_nginx_config=false
  if [[ -f "${nginx_conf}/nginx.conf" ]]; then
    if [[ "$CREATE_SERVICES" == "overwrite" ]]; then
      echo "  nginx config exists, overwrite it"
      write_nginx_config=true
    else
      echo "  nginx config exists, do not overwrite it"
      write_nginx_config=false
    fi
  else
    echo "  nginx config is not present, create it"
    write_nginx_config=true
  fi

  if [[ "$write_nginx_config" == "true" ]]; then
    mkdir -p /etc/nginx
    echo '
worker_processes auto;
pid /run/nginx.pid;
error_log /var/log/nginx/error.log;
include /etc/nginx/modules-enabled/*.conf;
events {
  worker_connections 256;
}
http {
  server {
    listen 80;
    client_max_body_size 8G;
    keepalive_timeout 1h;

    include /etc/nginx/conf.d/bodhi/bodhi-apps.conf;
    include /etc/nginx/conf.d/bodhi/bodhi-control.conf;

    error_page 503 /503.html;
    location = /503.html {
      root /etc/nginx/conf.d/bodhi/html;
      internal;
    }

  }
}
' > "${nginx_conf}/nginx.conf"

    mkdir -p /etc/nginx/conf.d/bodhi/switch

    echo '
location /control {
	proxy_read_timeout 12h;

  proxy_bind 127.0.0.1;
  proxy_pass http://127.0.0.1:6800;
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
}

' > /etc/nginx/conf.d/bodhi/switch/bodhi-control-enabled.conf

    echo '
location /control {
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
  return 503;
}
' > /etc/nginx/conf.d/bodhi/switch/bodhi-control-disabled.conf

    echo '

location / {
  proxy_bind 127.0.0.1;
  proxy_pass http://127.0.0.1:2000;
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
}

location /ws {
  proxy_bind 127.0.0.1;
  proxy_pass http://127.0.0.1:5000;
  proxy_http_version 1.1;
  proxy_set_header Upgrade $http_upgrade;
  proxy_set_header Connection "upgrade";
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
}

location /oidc {
  proxy_bind 127.0.0.1;
  proxy_pass http://127.0.0.1:3000/oidc;
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
}

location /web {
  proxy_bind 127.0.0.1;
  proxy_pass http://127.0.0.1:5041/web;
  proxy_set_header Host $host;
  proxy_set_header X-Real-IP $remote_addr;
  proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
  proxy_set_header X-Forwarded-Proto $scheme;
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
}
' > /etc/nginx/conf.d/bodhi/switch/bodhi-apps-enabled.conf

    echo '
location / {
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
  return 503;
}

location /ws {
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
  return 503;
}

location /oidc {
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
  return 503;
}

location /web {
  proxy_no_cache 1;
  proxy_cache_bypass 1;
  add_header Cache-Control "no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0";
  return 503;
}
' > /etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf

    mkdir -p /etc/nginx/conf.d/bodhi/html
    echo '
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Service Unavailable</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            text-align: center;
            padding: 50px;
        }
        h1 {
            font-size: 50px;
            color: #ff0000;
        }
        p {
            font-size: 20px;
        }
    </style>
</head>
<body>
    <h1>503 Service Unavailable</h1>
    <p>Service unavailable during maintenance - please try again later.</p>
</body>
</html>
' > /etc/nginx/conf.d/bodhi/html/503.html
  fi
}

create_services() {
  if [[ "$FRAMEWORK_LANGUAGE" == "nodejs" ]]; then
    delete_bodhi_control_service
  fi
  
  create_bodhi_apps_service
  
  if [[ "$FRAMEWORK_LANGUAGE" == "dotnet" ]]; then
    create_bodhi_control_service
    create_nginx_services
  fi
}

# we need this because the it seems django.service process is not stopping on service stop
stop_django_service_processes() {
  if [[ ! -f ${django_systemd} ]]; then
    echo "Django service file not found: ${django_systemd}"
    return
  fi
  local exec_start=$(get_ini_value "ExecStart" "Service" "$django_systemd")
  if [[ -z "$exec_start" ]]; then
    echo "ExecStart not found in ${django_systemd}"
    return
  fi

  #echo "   Stopping django.service processes..."
  for PID in $(ps ax -o pid,stat,command | grep -v grep | grep python | grep "$exec_start" | grep ' S[s|l] ' | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      #echo "    The python django.service process with PID $PID is running. Attempting to stop..."
      kill $PID
    fi
  done

  # loop and check every second if the process is still alive
  for i in {1..45}
  do
    local done=true
    for PID in $(ps ax | grep -v grep | grep python | grep "$exec_start" | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        done=false
        break
      fi
    done
   
    if [[ "${done}" == "true" ]]; then
      #echo "    All processes were killed"
      break
    else
      sleep 1
    fi
  done

  for PID in $(ps ax | grep -v grep | grep python | grep "$exec_start" | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      #echo "    The python django.service process with PID $PID is still running. Attempting to SIGKILL kill..."
      kill -9 $PID
    fi
  done
}

stop_pyrunner_service_processes() {
  # we need this because the pyrunner.service process is not stopping on service stop
  if [[ ! -f ${pyrunner_systemd} ]]; then
    echo "pyrunner service file not found: ${pyrunner_systemd}"
    return
  fi
  local exec_start=$(get_ini_value "ExecStart" "Service" "$pyrunner_systemd")
  if [[ -z "$exec_start" ]]; then
    echo "ExecStart not found in ${pyrunner_systemd}"
    return
  fi  
  
  #echo "   Stopping pyrunner.service processes..."
  for PID in $(ps ax -o pid,stat,command | grep -v grep | grep python | grep "$exec_start" | grep ' S[s|l] ' | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      #echo "    The python pyrunner.service process with PID $PID is running. Attempting to stop..."
      kill $PID
    fi
  done    
  # loop and check every second if the process is still alive
  for i in {1..45}
  do
    local done=true
    for PID in $(ps ax | grep -v grep | grep python | grep "$exec_start" | awk '{print $1}'); do
      if ps -p $PID > /dev/null 2>&1; then
        done=false
        break
      fi
    done  
    if [[ "${done}" == "true" ]]; then
      #echo "    All processes were killed"
      break
    else
      sleep 1
    fi
  done  
  for PID in $(ps ax | grep -v grep | grep python | grep "$exec_start" | awk '{print $1}'); do
    if ps -p $PID > /dev/null 2>&1; then
      #echo "    The python pyrunner.service process with PID $PID is still running. Attempting to SIGKILL kill..."
      kill -9 $PID
    fi
  done
}

# pyrunner.service is responsive but is a good idea to also ckeck
stop_pyrunner_services() {
  if [[ -f ${pyrunner_systemd} ]]; then
      if systemctl is-active --quiet pyrunner.service; then
        echo "  pyrunner service is running, stopping now..."
        systemctl stop pyrunner.service > /dev/null 2>&1
        sleep 5
      fi
      stop_pyrunner_service_processes
  fi

  if [[ -f ${django_systemd} ]]; then
      if systemctl is-active --quiet django.service; then
        echo "  pyrunner django service is running, stopping now..."
        systemctl stop django.service > /dev/null 2>&1
        sleep 15
      fi
      stop_django_service_processes
  fi

  if [[ -f ${chownwdog_systemd} ]]; then
      if systemctl is-active --quiet chownwdog.service; then
        echo "  chownwdog service is running, stopping now..."
        systemctl stop chownwdog.service > /dev/null 2>&1
      fi
  fi
}

start_pyrunner_services() {

  if [[ -f ${django_systemd} ]]; then
      if ! systemctl is-active --quiet django.service; then
        echo "  pyrunner django service is not running, starting now..."
        systemctl start django.service > /dev/null 2>&1
      fi
  fi

  if [[ -f ${pyrunner_systemd} ]]; then
      if ! systemctl is-active --quiet pyrunner.service; then
        echo "  pyrunner service is not running, starting now..."
        sleep 15      # wait for django to start
        systemctl start pyrunner.service > /dev/null 2>&1
      fi
  fi

  if [[ -f ${chownwdog_systemd} ]]; then
      if ! systemctl is-active --quiet chownwdog.service; then
        echo "  chownwdog service is not running, starting now..."
        systemctl start chownwdog.service > /dev/null 2>&1
      fi
  fi
}

generate_services_report() {
  local output_folder=$1
  local output_file=$2

  mkdir -p "${output_folder}/journalctl"

  echo "====================================================================" >> "${output_file}"
  echo "Services Info" >> "${output_file}"
  echo "-------------" >> "${output_file}"

  echo "bodhi-apps service status [ systemctl status bodhi-apps ]" >> "${output_file}"
  systemctl status bodhi-apps  >> "${output_file}"
  echo "please see the bodhi-apps service journal log in './journalctl/bodhi-apps.log' [journalctl -u bodhi-apps" >> "${output_file}"
  echo "" >> "${output_file}"
  local bodhiapps_journalctl_log="${output_folder}/journalctl/bodhi-apps.log"
  journalctl -u bodhi-apps >> "${bodhiapps_journalctl_log}"

  echo "bodhi-control service status [ systemctl status bodhi-control ]" >> "${output_file}"
  systemctl status bodhi-control  >> "${output_file}"
  echo "please see the bodhi-control service journal log in './journalctl/bodhi-control.log' [journalctl -u bodhi-control" >> "${output_file}"
  echo "" >> "${output_file}"
  local bodhicontrol_journalctl_log="${output_folder}/journalctl/bodhi-control.log"
  journalctl -u bodhi-control >> "${bodhicontrol_journalctl_log}"

  echo "nginx service status [ systemctl status nginx ]" >> "${output_file}"
  systemctl status nginx  >> "${output_file}"
  echo "please see the nginx service journal log in './journalctl/nginx.log' [journalctl -u nginx" >> "${output_file}"
  echo "" >> "${output_file}"
  local nginx_journalctl_log="${output_folder}/journalctl/nginx.log"
  journalctl -u nginx >> "${nginx_journalctl_log}"

  echo "postgresql service status [ systemctl status postgresql ]" >> "${output_file}"
  systemctl status postgresql  >> "${output_file}"
  echo "please see the postgresql service journal log in './journalctl/postgresql.log' [journalctl -u postgresql" >> "${output_file}"
  echo "" >> "${output_file}"
  local postgresql_journalctl_log="${output_folder}/journalctl/postgresql.log"
  journalctl -u postgresql >> "${postgresql_journalctl_log}"

  echo "" >> "${output_file}"
  echo "====================================================================" >> "${output_file}"
  echo "" >> "${output_file}"
}

enable_nginx_bodhi_proxy() {
  local apps=${1:-"yes"}
  local control=${2:-"yes"}
  local reload=${3:-"yes"}

  local changed=false
  if [[ "$apps" == "yes" ]]; then
    if [[ -f "/etc/nginx/conf.d/bodhi/switch/bodhi-apps-enabled.conf" ]]; then
      if [[ ! -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" ]]; then
        echo "  enabling bodhi-apps nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-apps-enabled.conf /etc/nginx/conf.d/bodhi/bodhi-apps.conf
        changed=true
      elif ! diff "/etc/nginx/conf.d/bodhi/switch/bodhi-apps-enabled.conf" "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" > /dev/null 2>&1; then
        # they are different
        echo "  enabling bodhi-apps nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-apps-enabled.conf /etc/nginx/conf.d/bodhi/bodhi-apps.conf
        changed=true
      fi
    else
      if [[ -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" ]]; then
        echo "  mising switch file to enable, removing bodhi-apps nginx proxy..."
        rm -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf"
        changed=true
      fi
    fi
  fi
  if [[ "$control" == "yes" ]]; then
    if [[ -f "/etc/nginx/conf.d/bodhi/switch/bodhi-control-enabled.conf" ]]; then
      if [[ ! -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf" ]]; then
        echo "  enabling bodhi-control nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-control-enabled.conf /etc/nginx/conf.d/bodhi/bodhi-control.conf
        changed=true
      elif ! diff "/etc/nginx/conf.d/bodhi/switch/bodhi-control-enabled.conf" "/etc/nginx/conf.d/bodhi/bodhi-control.conf" > /dev/null 2>&1; then
        # they are different
        echo "  enabling bodhi-control nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-control-enabled.conf /etc/nginx/conf.d/bodhi/bodhi-control.conf
        changed=true
      fi
    else
      if [[ -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf" ]]; then
        echo "  mising switch file to enable, removing bodhi-control nginx proxy..."
        rm -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf"
        changed=true
      fi
    fi
  fi

  if [[ "$reload" == "yes" ]] && [[ "$changed" == "true" ]]; then
    echo "  reloading the nginx configuration on enable bodhi..."
    nginx -s reload  > /dev/null 2>&1
  fi
}

disable_nginx_bodhi_proxy() {
  local apps=${1:-"yes"}
  local control=${2:-"yes"}
  local reload=${3:-"yes"}

  local changed=false
  if [[ "$apps" == "yes" ]]; then
    if [[ -f "/etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf" ]]; then
      if [[ ! -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" ]]; then
        echo "  disabling bodhi-apps nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf /etc/nginx/conf.d/bodhi/bodhi-apps.conf
        changed=true
      elif ! diff "/etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf" "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" > /dev/null 2>&1; then
        # they are different
        echo "  disabling bodhi-apps nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf /etc/nginx/conf.d/bodhi/bodhi-apps.conf
        changed=true
      fi
    else
      if [[ -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf" ]]; then
        echo "  mising switch file to disable, removing bodhi-apps nginx proxy..."
        rm -f "/etc/nginx/conf.d/bodhi/bodhi-apps.conf"
        changed=true
      fi
    fi
  fi
  if [[ "$control" == "yes" ]]; then
    if [[ -f "/etc/nginx/conf.d/bodhi/switch/bodhi-control-disabled.conf" ]]; then
      if [[ ! -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf" ]]; then
        echo "  disabling bodhi-control nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-control-disabled.conf /etc/nginx/conf.d/bodhi/bodhi-control.conf
        changed=true
      elif ! diff "/etc/nginx/conf.d/bodhi/switch/bodhi-control-disabled.conf" "/etc/nginx/conf.d/bodhi/bodhi-control.conf" > /dev/null 2>&1; then
        # they are different
        echo "  disabling bodhi-control nginx proxy..."
        cp -f /etc/nginx/conf.d/bodhi/switch/bodhi-control-disabled.conf /etc/nginx/conf.d/bodhi/bodhi-control.conf
        changed=true
      fi
    else
      if [[ -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf" ]]; then
        echo "  mising switch file to disable, removing bodhi-control nginx proxy..."
        rm -f "/etc/nginx/conf.d/bodhi/bodhi-control.conf"
        changed=true
      fi
    fi
  fi

  if [[ "$reload" == "yes" ]] && [[ "$changed" == "true" ]]; then
    echo "  reloading the nginx configuration on disable bodhi..."
    nginx -s reload  > /dev/null 2>&1
  fi
}

generate_services_health_report() {
  local output_folder=$1
  local output_file=$2

  echo "====================================================================" >> "${output_file}"
  echo "Services Health" >> "${output_file}"
  echo "---------------" >> "${output_file}"

  # Define service endpoints map
  local -A services_map=(
    [SimpleIdentityProvider Service]="http://127.0.0.1:3000"
    [UI Static Web Server]="http://127.0.0.1:2000"
    [API Service]="http://127.0.0.1:5041"
    [Control Service]="http://127.0.0.1:6800"
    [WebSocket Service]="http://127.0.0.1:5000"
  )

  if command -v curl > /dev/null 2>&1; then
    for service in "${!services_map[@]}"; do
      local endpoint="${services_map[$service]}/health"
      local http_response=$(curl -s -o /tmp/service_check_body -w "%{http_code}" --max-time 5 "$endpoint")
      local body_response=""
      if [[ -f /tmp/service_check_body ]]; then
        body_response=$(cat /tmp/service_check_body)
      fi
      rm -f /tmp/service_check_body
      if [[ "$body_response" == *"healthy"* ]]; then
        echo "bodhi $service: healthy [HTTP '$http_response', response: '$body_response' from '$endpoint']" >> "${output_file}"
      else
        echo "$service: NOT healthy [HTTP '$http_response', response: '$body_response' from '$endpoint']" >> "${output_file}"
      fi
    done

    echo "" >> "${output_file}"

    # Check pyReader service
    local pyreader_endpoint="http://127.0.0.1:8080"
    local pyreader_response=$(curl -s -o /tmp/pyreader_check_body -w "%{http_code}" --max-time 5 "$pyreader_endpoint")
    local pyreader_body=""
    if [[ -f /tmp/pyreader_check_body ]]; then
      pyreader_body=$(cat /tmp/pyreader_check_body)
    fi
    rm -f /tmp/pyreader_check_body
    if [[ "$pyreader_response" =~ ^2[0-9][0-9]$ ]] && [[ -n "$pyreader_body" ]] || [[ "$pyreader_response" == "404" ]]; then
      echo "PyReader/CalculationsEngine is up [HTTP '$pyreader_response' from '$pyreader_endpoint']" >> "${output_file}"
    else
      echo "PyReader/CalculationsEngine is NOT up [HTTP '$pyreader_response' from '$pyreader_endpoint']" >> "${output_file}"
    fi

    # Check PyRunner service
    local pyrunner_endpoint="http://127.0.0.1:8000"
    local pyrunner_response=$(curl -s -o /tmp/pyrunner_check_body -w "%{http_code}" --max-time 5 "$pyrunner_endpoint")
    local pyrunner_body=""
    if [[ -f /tmp/pyrunner_check_body ]]; then
      pyrunner_body=$(cat /tmp/pyrunner_check_body)
    fi
    rm -f /tmp/pyrunner_check_body
    if [[ "$pyrunner_response" =~ ^2[0-9][0-9]$ ]] && [[ -n "$pyrunner_body" ]] || [[ "$pyrunner_response" == "404" ]]; then
      echo "PyRunner is up [HTTP '$pyrunner_response' from '$pyrunner_endpoint']" >> "${output_file}"
    else
      echo "PyRunner is NOT up [HTTP '$pyrunner_response' from '$pyrunner_endpoint']" >> "${output_file}"
    fi

    echo "" >> "${output_file}"

    # Check if nginx is up by calling 127.0.0.1
    endpoint="http://127.0.0.1"
    local http_response=$(curl -s -o /tmp/nginx_check_body -w "%{http_code}" "$endpoint")
    local body_response=""
    if [[ -f /tmp/nginx_check_body ]]; then
      body_response=$(cat /tmp/nginx_check_body)
    fi
    rm -f /tmp/nginx_check_body
    if [[ "$http_response" =~ ^2[0-9][0-9]$ ]] && [[ -n "$body_response" ]]; then
      echo "nginx is up [HTTP '$http_response', response: '$body_response' from '$endpoint']" >> "${output_file}"
    elif [[ "$http_response" == "503" ]]; then
      echo "nginx is up [HTTP '$http_response', response: '$body_response' from '$endpoint']" >> "${output_file}"
      echo "" >> "${output_file}"
      echo "" >> "${output_file}"
      echo "WARNING: nginx is returning a '503 Service Unavailable' !" >> "${output_file}"
      echo "         this means that nginx is up but the application is in maintenance mode !" >> "${output_file}"
      echo "         please disable maintenance mode to have a fully functional application !" >> "${output_file}"
      echo "         for example run the powershell script on the laptop with reinitialize instrument true" >> "${output_file}"
      echo "         .\bodhi-apps.ps1 -targetMachine 192.168.0.2 -reinitializeInstrument \$true" >> "${output_file}"
      echo "         this will restart all services and disable maintenance mode [the database will NOT be affected]" >> "${output_file}"
      echo "         if you need assistance, please read the documentation and/or contact support" >> "${output_file}"
    else
      echo "nginx is NOT up [HTTP '$http_response', response: '$body_response' from '$endpoint']" >> "${output_file}"
    fi

    echo "" >> "${output_file}"

    # Check PostgreSQL health
    if command -v psql > /dev/null 2>&1; then
      # Try to connect to PostgreSQL and run a simple query
      local pg_result=$(sudo -u postgres psql -d postgres -c "SELECT 'healthy' as status;" -t -A 2>/dev/null)
      if [[ "$pg_result" == "healthy" ]]; then
        echo "postgresql is up and healthy [connection successful, query executed]" >> "${output_file}"
      else
        # Try alternative connection method if pg_isready is available
        if command -v pg_isready > /dev/null 2>&1; then
          local pg_status=$(sudo -u postgres pg_isready 2>/dev/null)
          if [[ $? -eq 0 ]]; then
            echo "postgresql is up but query failed [pg_isready: $pg_status]" >> "${output_file}"
          else
            echo "postgresql is NOT up [pg_isready failed: $pg_status]" >> "${output_file}"
          fi
        else
          echo "postgresql query failed and pg_isready not available [psql query result: '$pg_result']" >> "${output_file}"
        fi
      fi
    else
      echo "postgresql tools not available, skipping postgresql health check" >> "${output_file}"
    fi

    echo "" >> "${output_file}"
    
  else
    echo "curl not available, skipping health checks" >> "${output_file}"
  fi


  echo "" >> "${output_file}"
  echo "====================================================================" >> "${output_file}"
  echo "" >> "${output_file}"
}

