set_rtc() {

  # Check if the RTC device exists
  if [[ -e /dev/rtc ]] || [[ -e /dev/rtc0 ]]; then
      echo "  RTC device found"
  else
      echo "  RTC device not found"
      return 1
  fi

  # Check if the rtc_cmos module is loaded
  if ! lsmod | grep -q rtc_cmos; then
      echo "  RTC CMOS module is not loaded. Trying to load it..."
      modprobe rtc_cmos
      if ! lsmod | grep -q rtc_cmos; then
          echo "  failed to load RTC CMOS module"
          return 1
      fi
  fi

  # Check if hwclock can access the RTC
  if hwclock --verbose; then
      echo "  RTC is accessible, synchronizing RTC with system clock..."
      hwclock --systohc
      if [[ $? -eq 0 ]]; then
          echo "  RTC synchronized with system clock successfully"
      else
          echo "  failed to synchronize RTC with system clock"
      fi
  else
      echo "  RTC is not accessible"
      return 1
  fi
}

set_localization() {
  # we standardize the Pi to use:
  # en_US.UTF-8 as OS default locale and also for DB collation and ctype
  # UTF for time

  # Check if LOCALE_LANG locale is present
  local converted_locale=$(echo $LOCALE_LANG | sed -E 's/UTF-8/utf8/')
  local locale_present=$(locale -a | grep -c "${converted_locale}")

  local locale_changes=false
  if [[ $locale_present -eq 0 ]]; then
      echo ""
      echo "  locale ${LOCALE_LANG} not found, generating locale..."
      sed -i "s/^# *\(${LOCALE_LANG} UTF-8\)/\1/" /etc/locale.gen
      locale-gen ${LOCALE_LANG}
      sleep 3

      locale_changes=true
  fi

  # Check the current default locale
  local current_locale=$(locale | grep LANG | cut -d= -f2)

  if [[ "$current_locale" != "${LOCALE_LANG}" ]]; then
      echo ""
      echo "  current default locale is not ${LOCALE_LANG}, setting it as default..."
      update-locale LANG=${LOCALE_LANG}
      sleep 3
      source /etc/default/locale
      current_locale=$(locale | grep LANG | cut -d= -f2)
      echo "  current default locale is ${current_locale}"

      locale_changes=true
  fi

  # Get the current timezone
  local current_timezone=$(timedatectl | grep "Time zone" | awk '{print $3}')

  # Check if the current timezone is UTC
  if [[ "$current_timezone" != "${TIME_ZONE}" ]]; then
      echo ""
      echo "  current timezone is not ${TIME_ZONE}, setting system time to ${TIME_ZONE}..."
      timedatectl set-timezone ${TIME_ZONE}
      sleep 3
      current_timezone=$(timedatectl | grep "Time zone" | awk '{print $3}')
      if [[ "$current_timezone" == "${TIME_ZONE}" ]]; then
        echo "  successfully changed the current timezone to ${current_timezone}"
        set_rtc      
      else
        echo "  WARNING: failed to change the current timezone to ${current_timezone}"
      fi
  fi

  if [[ "$locale_changes" == "true" ]]; then
    echo "  there were locale changes, restart the postgresql service..."
    restart_postgres_service
  fi
}


generate_service_report_system_info() {
    local output_folder=$1
    local output_file=$2

    mkdir -p "${output_folder}/system"

    echo "====================================================================" >> "${output_file}"
    echo "System Info" >> "${output_file}"
    echo "-----------" >> "${output_file}"

    echo "hostname: $(hostname -f)" >> "${output_file}"
    echo "" >> "${output_file}"

    if command -v uname > /dev/null 2>&1; then
        echo "system information[ uname -a ]:" >> "${output_file}"
        uname -a >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v dpkg > /dev/null 2>&1; then
        echo "OS architeckture[ dpkg --print-architecture ]:" >> "${output_file}"
        dpkg --print-architecture  >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v getconf > /dev/null 2>&1; then
        echo "long integer number of bits[ getconf LONG_BIT ]:" >> "${output_file}"
        getconf LONG_BIT >> "${output_file}"

        echo "please see more details in the './system/getconf.log'" >> "${output_file}"
        echo "" >> "${output_file}"

        local getconf_log="${output_folder}/system/getconf.log"
        getconf -a >> "${getconf_log}"
    fi

    if command -v lsb_release > /dev/null 2>&1; then
        echo "Linux Standard Base (LSB) release[ lsb_release -a ]:" >> "${output_file}"
        lsb_release -a >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v vcgencmd > /dev/null 2>&1; then
        echo "videocore firmware version[ vcgencmd version ]:" >> "${output_file}"
        vcgencmd version  >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v df > /dev/null 2>&1; then
        echo "Disk space usage [df -h]:" >> "${output_file}"
        df -h  >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v locale > /dev/null 2>&1; then
        echo "Installed locale [locale -a]:" >> "${output_file}"
        locale -a  >> "${output_file}"
        echo "" >> "${output_file}"

        echo "Current selected locale [locale]:" >> "${output_file}"
        locale  >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    echo "Current date and time [date]:" >> "${output_file}"
    date >> "${output_file}"
    echo "" >> "${output_file}"

    if command -v timedatectl > /dev/null 2>&1; then
        echo "Current time [timedatectl]:" >> "${output_file}"
        timedatectl_output=$(timedatectl status)
        if [ -n "$timedatectl_output" ]; then
            echo "$timedatectl_output" >> "${output_file}"
        else
            # Check if systemd-timedated is running
            if ps aux | grep '[s]ystemd-timedated' > /dev/null; then
                echo "(timedatectl output empty, but systemd-timedated is running)" >> "${output_file}"
                ps aux | grep '[s]ystemd-timedated' >> "${output_file}"
            else
                echo "(timedatectl output empty, and systemd-timedated is NOT running)" >> "${output_file}"
            fi
        fi
        echo "" >> "${output_file}"
    fi

    if [ -f /etc/timezone ]; then
        echo "Current timezone file [/etc/timezone]:" >> "${output_file}"
        cat /etc/timezone >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    echo "Timezone symlink [/etc/localtime]:" >> "${output_file}"
    ls -l /etc/localtime >> "${output_file}"
    echo "" >> "${output_file}"

    if command -v hwclock > /dev/null 2>&1; then
        echo "Hardware clock [hwclock]:" >> "${output_file}"
        hwclock --verbose >> "${output_file}" 2>&1
        echo "" >> "${output_file}"
    fi
   
    echo "Real-Time Clock [/dev/rtc]:" >> "${output_file}"
    if [[ -e /dev/rtc ]] || [[ -e /dev/rtc0 ]]; then
        echo "RTC device found" >> "${output_file}"
    else
        echo "RTC device not found" >> "${output_file}"
    fi
    echo "" >> "${output_file}"

    if [[ -n "$CLIENT_DATETIME" ]]; then
        check_for_datetime_skew "$CLIENT_DATETIME" "$SERVER_TIMESTAMP" >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v free > /dev/null 2>&1; then
        echo "memory usage [free -h]:" >> "${output_file}"
        free -h >> "${output_file}"
        echo "please see more details in the './system/memory.log'" >> "${output_file}"
        echo "" >> "${output_file}"

        local memory_log="${output_folder}/system/memory.log"
        echo "memory usage [free -h]:" >> "${memory_log}"
        free -h >> "${memory_log}"
        echo "" >> "${memory_log}"
        echo "memory usage [cat /proc/meminfo]:" >> "${memory_log}"
        cat /proc/meminfo >> "${memory_log}"
        echo "" >> "${memory_log}"
    fi

    if command -v ps > /dev/null 2>&1; then
        echo "current running processes:" >> "${output_file}"
        ps -eo user,pid,%cpu,%mem,rss,args | grep -E '%CPU %MEM|MMPR|postgresql|node|nginx|python' | grep -v 'grep' >> "${output_file}"
        echo "please see more details in the './system/processes.log'" >> "${output_file}"
        echo "" >> "${output_file}"

        local processes_log="${output_folder}/system/processes.log"
        ps aux | grep -v 'grep' | grep -v 'bodhi-apps.sh' >> "${processes_log}"
        echo "" >> "${processes_log}"
    fi

    if command -v ip > /dev/null 2>&1; then
        echo "network interfaces [ip link]" >> "${output_file}"
        ip link >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v ifconfig > /dev/null 2>&1; then
        echo "network interfaces [ifconfig]" >> "${output_file}"
        ifconfig >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v netstat > /dev/null 2>&1; then
        echo "tcp ports in LISTEN mode [netstat -tlnp]:" >> "${output_file}"
        netstat -tlnp | grep LISTEN >> "${output_file}"
        echo "please see more details in the './system/ports.log'" >> "${output_file}"
        echo "" >> "${output_file}"

        local port_log="${output_folder}/system/ports.log"
        netstat >> "${port_log}"
        echo "" >> "${port_log}"
    fi

    if command -v iptables > /dev/null 2>&1; then
        echo "please see the iptables rules in './system/firewall.log'" >> "${output_file}"
        echo "" >> "${output_file}"
        local iptables_log="${output_folder}/system/iptables.log"
        echo "filter rules [ iptables -L -v -n ]" >> "${iptables_log}"
        echo "----------------------------------" >> "${iptables_log}"
        iptables -L -v -n >> "${iptables_log}"
        echo "" >> "${iptables_log}"
        echo "nat rules [ iptables -t nat -L -v -n ]" >> "${iptables_log}"
        echo "--------------------------------------" >> "${iptables_log}"
        iptables -t nat -L -v -n >> "${iptables_log}"
        echo "" >> "${iptables_log}"
        echo "mangle rules [ iptables -t mangle -L -v -n ]" >> "${iptables_log}"
        echo "--------------------------------------------" >> "${iptables_log}"
        iptables -t mangle -L -v -n >> "${iptables_log}"
        echo "" >> "${iptables_log}"
    fi

    if command -v nft > /dev/null 2>&1; then
        echo "please see the nft rules in './system/nft.log'" >> "${output_file}"
        echo "" >> "${output_file}"
        local nft_log="${output_folder}/system/nft.log"
        nft list ruleset >> "${nft_log}"
    fi

    if command -v dmesg > /dev/null 2>&1; then
        local dmesg_log="${output_folder}/system/dmesg.log"
        echo "please see all the kernel ring buffer message in './system/dmesg.log' [dmesg -T]" >> "${output_file}"
        echo "" >> "${output_file}"
        dmesg -T >> "${dmesg_log}"

        local dmesg_can_log="${output_folder}/system/dmesg_can.log"
        echo "please see all the CAN kernel ring buffer message in './system/dmesg_can.log' [dmesg -T | grep can]" >> "${output_file}"
        echo "" >> "${output_file}"
        dmesg -T | grep can >> "${dmesg_can_log}"
    fi

    echo "python" >> "${output_file}"
    echo "------" >> "${output_file}"

    if command -v python > /dev/null 2>&1; then
        echo "python version [python --version]" >> "${output_file}"
        which python >> "${output_file}"
        python --version >> "${output_file}"
        echo "" >> "${output_file}"
    else
        echo "python does not exist" >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v python3 > /dev/null 2>&1; then
        echo "python3 version [python3 --version]" >> "${output_file}"
        which python3 >> "${output_file}"
        python3 --version >> "${output_file}"
        echo "" >> "${output_file}"
    else
        echo "python3 does not exist" >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    if command -v python3.11 > /dev/null 2>&1; then
        echo "python3.11 version [python3.11 --version]" >> "${output_file}"
        which python3.11 >> "${output_file}"
        python3.11 --version >> "${output_file}"
        echo "" >> "${output_file}"
    else
        echo "python3.11 does not exist" >> "${output_file}"
        echo "" >> "${output_file}"
    fi

    echo "/home users and disk space usage [du -sh]:" >> "${output_file}"
    for dir in /home/*; do
        if [[ -d "$dir" ]]; then
            # Get the username from the directory name
            user=$(basename "$dir")
            # Get the size of the directory
            size=$(du -sh "$dir" | cut -f1)
            # Print the username and the size of their home directory
            echo "User: $user, Size: $size" >> "${output_file}"
        fi
    done

    echo "" >> "${output_file}"
    echo "====================================================================" >> "${output_file}"
    echo "" >> "${output_file}"
}

generate_service_report_nginx() {
    local output_folder=$1
    local output_file=$2

    mkdir -p "${output_folder}/configuration/nginx"

    echo "====================================================================" >> "${output_file}"
    echo "nginx configuration" >> "${output_file}"
    echo "-------------------" >> "${output_file}"

    if [[ -f /etc/nginx/nginx.conf ]]; then
        echo "nginx main configuration file found [/etc/nginx/nginx.conf]:" >> "${output_file}"
        echo "please see a copy of the nginx main configuration file in './configuration/nginx/nginx.conf'" >> "${output_file}"
        cp -f /etc/nginx/nginx.conf "${output_folder}/configuration/nginx"
    else
        echo "nginx main configuration file [/etc/nginx/nginx.conf] not found" >> "${output_file}"
    fi
    
    if [[ -d /etc/nginx/conf.d ]]; then
        echo "nginx conf.d folder found [/etc/nginx/conf.d]:" >> "${output_file}"

        # Check for nginx bodhi config folder
        if [ ! -d /etc/nginx/conf.d/bodhi ]; then
            echo "WARNING: /etc/nginx/conf.d/bodhi folder does not exist" >> "${output_file}"
        else
            # Compare bodhi-apps.conf and bodhi-apps-disabled.conf
            if [ -f /etc/nginx/conf.d/bodhi/bodhi-apps.conf ] && [ -f /etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf ]; then
                if cmp -s /etc/nginx/conf.d/bodhi/bodhi-apps.conf /etc/nginx/conf.d/bodhi/switch/bodhi-apps-disabled.conf; then
                    echo "==================================================================" >> "${output_file}"
                    echo "WARNING: bodhi apps is in maintenance mode !!" >> "${output_file}"
                    echo "a 503 page will be served by nginx, with this content:" >> "${output_file}"
                    echo "Service unavailable during maintenance - please try again later." >> "${output_file}"
                    echo "all bodhi services proxied by nginx are unreachable even if they are up and running" >> "${output_file}"
                    echo "==================================================================" >> "${output_file}"
                fi
            else
                echo "WARNING: bodhi apps nginx configuration files are missing" >> "${output_file}"
            fi
        fi

        echo "please see a copy of the nginx conf.d folder in './configuration/nginx/conf.d'" >> "${output_file}"
        cp -rf /etc/nginx/conf.d "${output_folder}/configuration/nginx"
    else            
        echo "nginx conf.d folder [/etc/nginx/conf.d] not found" >> "${output_file}"
    fi

    # Copy all nginx logs to output_folder/log/nginx
    if [ -d /var/log/nginx ]; then
        mkdir -p "${output_folder}/log/nginx"
        cp -rf /var/log/nginx/* "${output_folder}/log/nginx/"
    else
        echo "nginx log folder [/var/log/nginx] not found" >> "${output_file}"
    fi
    echo "please see the nginx log files attached to this report in the './log' folder" >> "${output_file}"

    echo "" >> "${output_file}"
    echo "====================================================================" >> "${output_file}"
    echo "" >> "${output_file}"
}
