import os
import time
import logging
from glob import glob
from datetime import datetime, timedelta
from django.utils.deprecation import MiddlewareMixin
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error
from calculation_methods.py_calculations.calculation_constants.constants import LOG_DIR

def create_log_directory():
    #Ensure the log directory exists.
    if not os.path.exists(LOG_DIR):
        os.makedirs(LOG_DIR)
    return LOG_DIR

class LogCleanupMiddleware(MiddlewareMixin):
    
    # Class variable to track last cleanup time
    _last_cleanup_time = None
    _cleanup_interval = 6 * 60 * 60  # 6 hours in seconds
    
    def delete_old_logs(self):
        """Deletes old log files, keeping only today's log file."""
        logger = logging.getLogger('log_info')
        today = datetime.now().strftime('%Y-%m-%d')
        today_debug_file = f'debug_info_{today}.log'

        logs_deleted = False  # Track if any logs were deleted
        
        # First, clean up any leftover .deleting files from previous failed attempts
        for deleting_file in glob(os.path.join(LOG_DIR, "debug_info_*.log.deleting")):
            try:
                os.remove(deleting_file)
                logger.info(f"Cleaned up leftover temp file: {deleting_file}")
                logs_deleted = True
            except OSError as e:
                logger.warning(f"Failed to clean up temp file {deleting_file}: {e}")
        
        # Iterate only over files starting with 'debug_info_'
        for log_file in glob(os.path.join(LOG_DIR, "debug_info_*.log")):
            file_name = os.path.basename(log_file)
            if file_name != today_debug_file:
                try:
                    # Check if file exists and is not currently being written to
                    if os.path.exists(log_file):
                        # Try to rename first (Windows-specific check for file locks)
                        temp_name = log_file + ".deleting"
                        try:
                            os.rename(log_file, temp_name)
                            os.remove(temp_name)
                            logs_deleted = True  # Set True if a file is deleted
                            logger.info(f"Successfully deleted old log file: {log_file}")
                        except (OSError, PermissionError) as rename_error:
                            # If rename failed, file is likely locked - skip it
                            logger.warning(f"Log file {log_file} is in use, skipping deletion: {rename_error}")
                            # If temp file was created, try to rename it back
                            if os.path.exists(temp_name):
                                try:
                                    os.rename(temp_name, log_file)
                                except OSError:
                                    pass  # If this fails, temp file will be cleaned up next time
                except Exception as e:
                    # Catch any other unexpected exceptions and log them
                    logger.error(f"Unexpected error deleting log file {log_file}: {e}")
        
        return logs_deleted  # Return True if any log was deleted, else False  
    
    def process_request(self, request):
        """Run log cleanup only every 6 hours to avoid performance impact."""
        
        current_time = time.time()
        
        # Check if we need to run cleanup (first time or 6+ hours since last cleanup)
        if (self._last_cleanup_time is None or 
            current_time - self._last_cleanup_time >= self._cleanup_interval):
            
            try:
                self.delete_old_logs()
                # Update the last cleanup time only if successful
                LogCleanupMiddleware._last_cleanup_time = current_time
            except Exception as e:
                # Log the error but don't crash the request
                logger = logging.getLogger('log_info')
                logger.error(f"Log cleanup failed during middleware processing: {e}")
        
        return None  # Continue processing the request
