# calculation_factory.py
import numpy as np
from calculation_methods.py_calculations.strategies import BasicOperationStrategy, StatisticStrategy, CVStatisticStrategy
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations import calculation_utils as utils


class CalculationFactory:
    # Factory for creating calculation strategies
    BASIC_OPERATIONS = {
        '+': lambda x, y: x + y,  # Addition
        '-': lambda x, y: x - y,  # Subtraction
        '*': lambda x, y: x * y,  # Multiplication
        '/': lambda x, y: "undefined" if y == 0 else x / y  # Division
    }
    
    STATISTICAL_OPERATIONS = {
        'mean': StatisticStrategy(lambda x: np.nanmean(x)),  # Mean
        'median': StatisticStrategy(lambda x: np.nanmedian(x)),  # Median
        'std': StatisticStrategy(lambda x: np.nanstd(x)),  # Standard deviation
        'cv': CVStatisticStrategy(),  # Coefficient of variation
        'sum': StatisticStrategy(lambda x: np.nansum(x)),  # Sum
        'max': StatisticStrategy(lambda x: np.nanmax(x)),  # Maximum
        'min': StatisticStrategy(lambda x: np.nanmin(x))  # Minimum
    }
    
    @staticmethod
    def create_basic_calculation(operator):
        # Create strategy for basic operation
        operation = CalculationFactory.BASIC_OPERATIONS.get(operator)
        if not operation:
            utils.handle_error(ErrorMessages.ERROR_INVALID_OPERATOR)
        return BasicOperationStrategy(operation)
    
    @staticmethod
    def create_statistical_calculation(metric):
        # Create strategy for statistical calculation
        strategy = CalculationFactory.STATISTICAL_OPERATIONS.get(metric)
        if not strategy:
            utils.handle_error(ErrorMessages.ERROR_STATISTICAL_CALCULATION)
        return strategy