import numpy as np
from calculation_methods.py_calculations.calculation_constants.constants import FIVEPLMODEL
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations import calculation_utils as utils

class FiveParameterLogisticRegression:
    def __init__(self, input_data, standard_values, blank_values, weighted=False, forced_to_zero=False):
        # Store input data and options
        self.input_data = input_data
        self.standard_values = standard_values
        self.blank_values = blank_values
        self.weighted = weighted
        self.forced_to_zero = forced_to_zero
        
        # Process input data into averages and lists
        self.processed_data = utils.process_input_data(input_data)
        
        # Validate and convert x, y to numpy arrays, optionally compute weights
        self.x_values, self.y_values, self.weights = utils.validate_points(self.processed_data['x'], self.processed_data['y'], weighted)
        
        # Ensure x_values, y_values, and weights have the same length
        min_length = min(len(self.x_values), len(self.y_values))
        self.x_values = self.x_values[:min_length]
        self.y_values = self.y_values[:min_length]
        if self.weights is not None:
            self.weights = self.weights[:min_length]
        
        # Check for minimum 5 points required for 5PL fitting
        if len(self.x_values) < 5:
            utils.handle_error(ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS)
        
        # Fit the 5PL model using lmfit and store parameters
        self.params = utils.calculate_params_with_lmfit(x_values=self.x_values, y_values=self.y_values, model_type=FIVEPLMODEL, weighted=self.weighted, forced_to_zero=self.forced_to_zero, weights=self.weights)
        
        # Extract fitted parameters into a dictionary
        self.coefficients = self.extract_coefficients()

    def extract_coefficients(self):
        """
        Extract 5PL parameters as a single dictionary.

        Returns:
            dict: {'a': lower asymptote, 'b': slope, 'c': inflection, 'd': upper asymptote, 'g': asymmetry}
        """
        try:
            # Ensure params is a list with 5 elements
            if not isinstance(self.params, list):
                utils.handle_error(ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS)
            
            # Unpack parameters, setting a=0 if forced_to_zero
            a, b, c, d, g = (0, *self.params[1:]) if self.forced_to_zero else self.params
            
            # Return as a dictionary with float values
            return {'a': float(a), 'b': float(b), 'c': float(c), 'd': float(d), 'g': float(g)}
        except Exception as e:
            utils.handle_error(ErrorMessages.ERROR_GET_COEFFICIENTS.format(str(e)))

    def predict(self, x_values):
        #Predict y-values for given x-values using the 5PL model.
        
        # Handle empty or None input
        if x_values is None or len(x_values) == 0:
            return np.array([])
        
        # Use utility function to compute y-values with 5PL equation
        return utils.calculate_curve_y(np.array(x_values), FIVEPLMODEL, self.coefficients)

    def calculate_concentration(self, y):
        """
        Calculate concentration (x) from a y-value using the inverse 5PL equation.

        Formula: x = c * (((a - d) / (y - d))^(1/g) - 1)^(1/b)
        - a: Lower asymptote
        - d: Upper asymptote
        - c: Inflection point
        - b: Hill slope
        - g: Asymmetry factor

        """
        try:
            # Extract parameters from coefficients
            a, b, c, d, g = (self.coefficients[k] for k in ['a', 'b', 'c', 'd', 'g'])
            
            # Check for invalid conditions
            if not np.isfinite(y) or (y - d) == 0 or (a - d) == 0:
                return None
            
            # Compute inner term: (a - d) / (y - d)
            inner_term = (a - d) / (y - d)
            if inner_term <= 0:
                return None  # Avoid negative or zero base
            
            # Apply asymmetry factor g: ((a - d) / (y - d))^(1/g)
            base = (inner_term ** (1 / g)) - 1
            if base <= 0:
                return None  # Avoid negative or zero base
            
            # Compute concentration: c * base^(1/b)
            concentration = c * (base ** (1 / b))
            
            # Return only if finite and non-negative
            return concentration if np.isfinite(concentration) and concentration >= 0 else None
        except (OverflowError, ZeroDivisionError, ValueError):
            return None  # Handle numerical errors

    def calculate_cv(self, sample_type):
        #Calculate coefficient of variation for a sample type.
        
        # Compute concentrations for all y-values of the sample type
        concentrations = [self.calculate_concentration(y) for entry in self.input_data if entry['identifier'].startswith(sample_type) for y in entry['y']]
        
        # Filter out None values and compute CV
        return utils.compute_statistical_cv([c for c in concentrations if c is not None])

    def get_metrics(self):
        #Calculate and return all metrics for the 5PL model.

        # Predict y-values for the fitted x-values
        y_pred = self.predict(self.x_values)
        
        # Extract coefficients as a list for statistics calculation
        coeff_list = [self.coefficients[k] for k in ['a', 'b', 'c', 'd', 'g']]
        
        # Compute statistical metrics (R², MSE, etc.)
        metrics = utils.calculate_statistics(y_true=self.y_values, y_pred=y_pred, coefficients=coeff_list)
 
        # Calculate lower limit of detection
        metrics['LLD'] = utils.calculate_lld(blank_y=self.processed_data['blank_y'], model_type=FIVEPLMODEL, params=self.coefficients, x_values=self.x_values)
        
        # Generate table with sample group details
        additional_table_details = utils.generate_table_by_sample_groups(input_data=self.input_data, model_type=FIVEPLMODEL, params=self.coefficients, predict_func=self.calculate_concentration, cv_func=self.calculate_cv)
        
        # Generate interpolated curve points for plotting
        curve_data_points = utils.generate_interpolated_curve_data(x_data=self.x_values, y_data=self.y_values, curve_type=FIVEPLMODEL, coefficients=self.coefficients)
        
        # Return all results in a structured dictionary
        return {
            "method": FIVEPLMODEL,
            "function": self.coefficients,
            "metrics": metrics,
            "Additional_Table_Details": additional_table_details,
            "Curve_Data_Points": curve_data_points
        }