import collections.abc
from concurrent.futures import ThreadPoolExecutor, as_completed
import json
from logging import log
import logging
import numbers
import numpy as np

from django.http import JsonResponse
from calculation_methods.py_calculations.calculation_utils import extract_data_sets, process_statistic_data_sets
from calculation_methods.py_calculations.five_pl_module import FiveParameterLogisticRegression
from calculation_methods.py_calculations.four_pl_module import FourParameterLogisticRegression
from calculation_methods.py_calculations.basic_calculations import BasicCalculations
from calculation_methods.py_calculations.blank_corrections import BlankCorrections
from calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.linear_regression import LinearRegression
from calculation_methods.py_calculations.cubic_spline import CubicSplineInterpolator
from calculation_methods.py_calculations.response_handling.response_handler import ResponseHandler
from calculation_methods.py_calculations.response_handling.validations import InputValidator
from calculation_methods.py_calculations.response_handling.response_utils import ResponseUtils
import logging
log = logging.getLogger('log_info')
from calculation_methods.py_calculations.calculation_utils import handle_error
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

from functools import wraps
from calculation_methods.delete_logfiles import LogCleanupMiddleware



def basic_calculation(request):
    if request.method == 'POST':
        try:
            # Parse request data into a dictionary
            data = json.loads(request.body)
            request_id = data.get('id')

            # Extract parameters
            params = data.get('params', {})
            data_sets = params.get('data_sets')
            operator = params.get('operator')

            if not data_sets:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )
            if not operator:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_OPERATOR,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_OPERATOR)
            )
            x = []
            y = []
            for data_set in data_sets:
                operator_id = data_set.get('id')
                data = data_set.get('data')
                if isinstance(data, numbers.Number):
                    if operator_id == 'x':
                        x = data
                    else:
                        y = data
                else:
                    for data_set_element in data:
                        measurements = data_set_element.get('y')
                        arr = np.array(measurements)
                        if operator_id == 'x':
                            x = np.concatenate((x, arr.flatten()))
                        else:
                            y = np.concatenate((y, arr.flatten()))
                    if isinstance(x, np.ndarray):
                        x = x.tolist()
                    if isinstance(y, np.ndarray):
                        y = y.tolist()

            # Validate the input data
            InputValidator.validate_basic_calculation_inputs(x, y, operator)

            # Execute the calculation
            calculations = BasicCalculations(x, y, operator)
            result = calculations.basic_calculations()

            # Assign the calculation result to a response payload and return it
            result_set = {'id': '', 'data': []}
            if isinstance(result, (numbers.Number, str, float, int)):
                result_set['data'] = [result]
            else:
                for data_set in data_sets:
                    if not isinstance(data_set['data'], numbers.Number):
                        result_set['data'] = data_set['data']
                        for i in range(len(result_set['data'])):
                            for j in range(len(result_set['data'][i]['y'])):
                                if isinstance(result_set['data'][i]['y'][j], collections.abc.Sequence):
                                    for k in range(len(result_set['data'][i]['y'][j])):
                                        result_value = result.pop(0) if result.__len__() else None
                                        result_set['data'][i]['y'][j][k] = result_value
                                else:
                                    result_value = result.pop(0) if result.__len__() else None
                                    result_set['data'][i]['y'][j] = result_value
                        break
            log.info(f"Input data: {data_sets}, Result: {result_set}")
            return JsonResponse(ResponseHandler.success_response(request_id, result_set))

        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_INVALID_REQUEST)
        except json.JSONDecodeError as e:
            request_id = None
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except Exception as e:
            return ResponseUtils.handle_general_error(request_id, e, ErrorMessages.ERROR_INVALID_REQUEST)
    else:
        return ResponseUtils.handle_method_not_allowed()
def blank_correction(request):
    if request.method == 'POST':
        try:
            # Parse request data into a dictionary
            data = json.loads(request.body)
            request_id = data.get('id')

            # Extract parameters
            params = data.get('params', {})
            data_sets = params.get('data_sets')

            if not data_sets:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            blank_values = []
            input_data = []
            for data_set in data_sets:
                operator_id = data_set.get('id')
                data = data_set.get('data')
                for data_set_element in data:
                    measurements = data_set_element.get('y')
                    arr = np.array(measurements)
                    if operator_id == 'input_data':
                        input_data = np.concatenate((input_data, arr.flatten()))
                    else:
                        blank_values = np.concatenate((blank_values, arr.flatten()))
            # Convert numpy array to plain list, if applicable
            input_data = input_data.tolist() if isinstance(input_data, np.ndarray) else input_data
            blank_values = blank_values.tolist() if isinstance(blank_values, np.ndarray) else blank_values

            # Validate the input data
            InputValidator.validate_blank_correction_inputs(blank_values, input_data)

            # Execute the calculation
            blank_corrections = BlankCorrections(blank_values, input_data)
            result = blank_corrections.blank_correction()

            # Assign the calculation result to a response payload and return it
            result_set = {'id': '', 'data': []}
            for data_set in data_sets:
                if data_set['id'] == 'input_data':
                    result_set['id'] = data_set['id']
                    result_set['data'] = data_set['data']
                    for i in range(len(result_set['data'])):
                        for j in range(len(result_set['data'][i]['y'])):
                            if isinstance(result_set['data'][i]['y'][j], collections.abc.Sequence):
                                for k in range(len(result_set['data'][i]['y'][j])):
                                    result_value = result.pop(0) if result.__len__() else None
                                    result_set['data'][i]['y'][j][k] = result_value
                            else:
                                result_value = result.pop(0) if result.__len__() else None
                                result_set['data'][i]['y'][j] = result_value

            log.info(f"Input data: {data_sets}, Result: {result_set}")
            return JsonResponse(ResponseHandler.success_response(request_id, result_set))

        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_INVALID_REQUEST)
        except json.JSONDecodeError as e:
            request_id = None
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except Exception as e:
            return ResponseUtils.handle_general_error(request_id, e, ErrorMessages.ERROR_INVALID_REQUEST)
    else:
        return ResponseUtils.handle_method_not_allowed()

def statistics(request):
    if request.method == 'POST':
        try:
            # Parse request data into a dictionary
            data = json.loads(request.body)
            request_id = data.get('id')
            request_method = data.get('method')
            params = data.get('params', {})

             # Decide target key based on method
            target_source = 'sample_type' if request_method == 'statistics' else 'well'

            # Fetch data_sets based on request_method
            if request_method == 'statistics':
                data_sets = params.get('data_sets', [])
            elif request_method == 'kinetic_statistics':
                raw = params.get('data_sets', [])
                if not isinstance(raw, list) or len(raw) != 1 or 'data' not in raw[0]:
                    return handle_error(
                        ErrorMessages.ERROR_INVALID_REQUEST,
                        response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                    )
                data_sets = raw[0]['data']
            else:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )
            
            metrics = params.get('metrics')

            # Validate that 'data_sets' and 'metrics' are provided
            if not data_sets or not metrics:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                    # Uses JsonRpcCode -32600 (Invalid Request)
                )

            # Initialize result container
            result = {'metrics': []}

            # Process data sets concurrently
            with ThreadPoolExecutor() as executor:
                futures = []
                for data_set in data_sets:
                    futures.append(executor.submit(process_statistic_data_sets, data_set, request_method, target_source))

                for future in as_completed(futures):
                    try:
                        sample_type_data = future.result()
                        InputValidator.validate_statistics_inputs(sample_type_data['data'])
                        sample_type_statistics = SampleTypeStatistics(sample_type_data['data'], metrics)
                        metric = sample_type_statistics.calculate_statistics()
                        metric['label'] = sample_type_data[target_source]
                        result['metrics'].append(metric)
                    except ValueError as e:
                        return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_STATISTICAL_CALCULATION)
                    except Exception as e:
                        return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_STATISTICAL_CALCULATION)
                    
            log.info(f"Input data: {data_sets}, Result: {result}")
            return JsonResponse(ResponseHandler.success_response(request_id, result))

        except json.JSONDecodeError as e:
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_STATISTICAL_CALCULATION)
        except Exception as e:
            return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_INTERNAL_SERVER)
    else:
        return ResponseUtils.handle_method_not_allowed()
    
def linear_regression(request):
    if request.method == 'POST':
        try:
            # Parse request data into a dictionary
            data = json.loads(request.body)
            request_id = data.get('id')

            try:
                # Extract parameters
                params = data.get('params', {})
                data_sets = params.get('data_sets')
                weighted = params.get('weighted', False)
                forced_to_zero = params.get('forced_to_zero', False)

                # Validate that 'data_sets' is not empty
                if not data_sets:
                    return handle_error(
                        ErrorMessages.ERROR_INVALID_REQUEST,
                        response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                    )

                input_data, standard_values, blank_values = extract_data_sets(data_sets)

                # Execute the calculation
                linear_regression_instance = LinearRegression(input_data, standard_values, blank_values, weighted, forced_to_zero)
                result = linear_regression_instance.execute()

                # Assign the calculation result to a response payload and return it
                log.info(f"Input data: {data_sets}, Result: {result}")
                return JsonResponse(ResponseHandler.success_response(request_id, result))

            except ValueError as e:
                return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_GET_COEFFICIENTS)
            except json.JSONDecodeError as e:
                return ResponseUtils.handle_json_decode_error(request_id, e)
            except Exception as e:
                return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_INTERNAL_SERVER)
        except json.JSONDecodeError as e:
            return ResponseUtils.handle_json_decode_error(request_id, e)
    else:
        return ResponseUtils.handle_method_not_allowed()

def cubic_spline(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            request_id = data.get('id')
            params = data.get('params', {})
            if not isinstance(params, dict):
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            data_sets = params.get('data_sets', [])
            weighted = params.get('weighted', False)

            if not data_sets:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            input_data, standard_values, blank_values = extract_data_sets(data_sets)
            interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values, weighted)
            metrics = interpolator.get_coefficients()

            response_data = {
                'id': request_id,
                'result': metrics
            }
            log.info(f"Input data: {data_sets}, Result: {response_data}")
            return JsonResponse(response_data)

        except json.JSONDecodeError as e:
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_GET_COEFFICIENTS)
        except Exception as e:
            return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_INTERNAL_SERVER)
    else:
        return ResponseUtils.handle_method_not_allowed()

def fourpl_calculation(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            request_id = data.get('id')
            params = data.get('params', {})

            if not isinstance(params, dict):
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            data_sets = params.get('data_sets', [])
            weighted = params.get('weighted', False)
            forced_to_zero = params.get('forced_to_zero', False)

            if not data_sets:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            input_data, standard_values, blank_values = extract_data_sets(data_sets)
            interpolator = FourParameterLogisticRegression(input_data, standard_values, blank_values, weighted, forced_to_zero)
            metrics = interpolator.get_metrics()

            response_data = {
                'id': request_id,
                'result': metrics
            }
            log.info(f"Input data: {data_sets}, Result: {response_data}")
            return JsonResponse(response_data)

        except json.JSONDecodeError as e:
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_MODEL_CALCULATION)
        except Exception as e:
            return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_INTERNAL_SERVER)
    else:
        return ResponseUtils.handle_method_not_allowed()

def fivepl_calculation(request):
    if request.method == 'POST':
        try:
            data = json.loads(request.body)
            request_id = data.get('id')
            params = data.get('params', {})

            if not isinstance(params, dict):
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            data_sets = params.get('data_sets', [])
            weighted = params.get('weighted', False)
            forced_to_zero = params.get('forced_to_zero', False)

            if not data_sets:
                return handle_error(
                    ErrorMessages.ERROR_INVALID_REQUEST,
                    response=ResponseHandler.invalid_request(request_id, ErrorMessages.ERROR_INVALID_REQUEST)
                )

            input_data, standard_values, blank_values = extract_data_sets(data_sets)
            interpolator = FiveParameterLogisticRegression(input_data, standard_values, blank_values, weighted, forced_to_zero)
            metrics = interpolator.get_metrics()

            response_data = {
                'id': request_id,
                'result': metrics
            }
            log.info(f"Input data: {data_sets}, Result: {response_data}")
            return JsonResponse(response_data)

        except json.JSONDecodeError as e:
            return ResponseUtils.handle_json_decode_error(request_id, e)
        except ValueError as e:
            return ResponseUtils.handle_value_error(request_id, e, ErrorMessages.ERROR_MODEL_CALCULATION)
        except Exception as e:
            return ResponseUtils.handle_internal_error(request_id, e, ErrorMessages.ERROR_INTERNAL_SERVER)
    else:
        return ResponseUtils.handle_method_not_allowed()
