import numpy as np
import numbers
import collections.abc
import logging

log = logging.getLogger('django')
from calculation_methods.py_calculations.calculation_utils import handle_error
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

class InputValidator:
    @staticmethod
    def validate_inputs(data):
        if (isinstance(data, collections.abc.Sequence) or
                (isinstance(data, np.ndarray) and np.issubdtype(data.dtype, np.number))):
            for item in data:
                if not isinstance(item, numbers.Number):
                    handle_error(ErrorMessages.ERROR_INVALID_REQUEST)
        elif not isinstance(data, numbers.Number):
            handle_error(ErrorMessages.ERROR_INVALID_REQUEST)

    @staticmethod
    def are_inputs_equal_length(list1, list2):
        if (isinstance(list1, list) and isinstance(list2, list)) and len(list1) != len(list2):
            handle_error(ErrorMessages.ERROR_LIST_LENGTH_DIFFERENT)

    @staticmethod
    def validate_division_by_zero(y, operator):
        if y == 0 and operator == "/":
            handle_error(ErrorMessages.ERROR_DIVISION_BY_ZERO)

    @staticmethod
    def validate_operator(operator):
        valid_operators = ['+', '-', '*', '/']
        if operator not in valid_operators:
            handle_error(ErrorMessages.ERROR_INVALID_OPERATOR, operator, valid_operators)

    @staticmethod
    def validate_basic_calculation_inputs(x, y, operator):
        InputValidator.validate_inputs(x)
        InputValidator.validate_inputs(y)
        InputValidator.are_inputs_equal_length(x, y)
        if isinstance(x, numbers.Number):
            InputValidator.validate_division_by_zero(x, operator)
        if isinstance(y, numbers.Number):
            InputValidator.validate_division_by_zero(y, operator)
        InputValidator.validate_operator(operator)

    @staticmethod
    def validate_blank_correction_inputs(blank_values, input_data):
        if not isinstance(blank_values, list):
            handle_error(ErrorMessages.ERROR_BLANK_VALUES_EMPTY)
        if not isinstance(input_data, list):
            handle_error(ErrorMessages.ERROR_MISSING_INPUT_DATA)

    @staticmethod
    def validate_linear_regression_inputs(input_values, standard_values, blank_values):
        InputValidator.validate_inputs(input_values)
        InputValidator.validate_inputs(standard_values)
        InputValidator.validate_inputs(blank_values)
        
    @staticmethod
    def validate_statistics_inputs(input_values):
        InputValidator.validate_inputs(input_values)