# strategies.py
import numpy as np
from calculation_methods.py_calculations.calculation_strategy import CalculationStrategy
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations import calculation_utils as utils

class BasicOperationStrategy(CalculationStrategy):
    # Strategy for basic arithmetic operations
    def __init__(self, operation_func):
        # Initialize with operation lambda
        self.operation = operation_func
    
    def calculate(self, x, y):
        # Perform operation on two values
        return self.operation(x, y)

class StatisticStrategy(CalculationStrategy):
    # Strategy for simple statistical calculations
    def __init__(self, statistic_func):
        # Initialize with NumPy statistic function
        self.statistic_func = statistic_func
    
    def calculate(self, values):
        # Calculate statistic on array
        try:
            return self.statistic_func(values)
        except Exception:
            utils.handle_error(ErrorMessages.ERROR_STATISTICAL_CALCULATION)
            return None

class CVStatisticStrategy(CalculationStrategy):
    # Strategy for coefficient of variation
    def calculate(self, values):
        # Calculate CV as (std/mean) * 100
        try:
            mean = np.nanmean(values)
            if mean == 0:
                return None  # Avoid division by zero
            return (np.nanstd(values) / mean) * 100
        except Exception:
            utils.handle_error(ErrorMessages.ERROR_STATISTICAL_CALCULATION)
            return None