import re
import pytest
import numpy as np
from calculation_methods.py_calculations.basic_calculations import BasicCalculations
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

@pytest.fixture
def basic_calculations():
    def _basic_calculations(x, y, operator):
        return BasicCalculations(x, y, operator)
    return _basic_calculations

class TestAddition:

    # Tests element-wise addition of two large arrays of random integers, ensuring scalability and accuracy.
    @pytest.mark.happy_path
    def test_addition_large_random_arrays(self, basic_calculations):
        # Arrange: Create two 10,000-element arrays of random integers (-1M to 1M), compute expected sums as strings.
        arr1 = np.random.randint(-1_000_000, 1_000_000, size=10_000)
        arr2 = np.random.randint(-1_000_000, 1_000_000, size=10_000)
        expected = (arr1 + arr2).astype(str).tolist()

        # Act: Perform element-wise addition using the Addition method.
        result = basic_calculations(arr1.tolist(), arr2.tolist(), '+').basic_calculations()

        # Assert: Verify the result matches the expected list of string sums, confirming large array handling.
        assert result == expected

    # Tests addition with arrays containing infinities, NaNs, and very large floats, ensuring special value handling.
    @pytest.mark.edge_case
    def test_addition_with_infinities_and_nans(self, basic_calculations):
        # Arrange: Create arrays with special values (inf, -inf, NaN) and large floats, compute expected results with NumPy.
        arr1 = np.array([np.inf, -np.inf, np.nan, np.random.uniform(1e307, 1e308), np.random.uniform(1e307, 1e308)])
        arr2 = np.array([np.random.uniform(1e306, 1e307), np.random.uniform(1e306, 1e307), 5, np.nan, np.random.uniform(1e307, 1e308)])
        expected = np.add(arr1, arr2, dtype=np.float64)  # Perform numpy addition

        # Act: Perform element-wise addition with special values.
        result = basic_calculations(arr1.tolist(), arr2.tolist(), '+').basic_calculations()
        
        # Convert result to NumPy array for comparison with NaN and inf support.
        result_np = np.array(result, dtype=np.float64)

        # Assert: Check that results match expected values or are both NaN where applicable, ensuring proper edge case handling.
        assert np.all((result_np == expected) | (np.isnan(result_np) & np.isnan(expected)))

    # Tests addition of high-precision floating-point numbers, ensuring accurate float arithmetic.
    @pytest.mark.happy_path
    def test_addition_high_precision_floats(self, basic_calculations):
        # Arrange: Generate two 5-element arrays of random floats (-10 to 10), compute expected sums as strings.
        arr1 = np.random.uniform(-10, 10, size=5).tolist()
        arr2 = np.random.uniform(-10, 10, size=5).tolist()
        expected = [str(a + b) for a, b in zip(arr1, arr2)]

        # Act: Perform element-wise addition of float arrays.
        result = basic_calculations(arr1, arr2, '+').basic_calculations()

        # Assert: Verify the result matches the expected list of string sums, confirming float precision.
        assert result == expected

    # Tests adding a scalar to a large flattened multi-dimensional array, ensuring scalar-array operation scalability.
    @pytest.mark.happy_path
    def test_addition_scalar_with_large_multidimensional_array(self, basic_calculations):
        # Arrange: Create a scalar (500K-1M) and a 50,000-element array (1K-10K), compute expected sums as strings.
        scalar = np.random.randint(500_000, 1_000_000)
        arr = np.random.randint(1_000, 10_000, size=50_000)
        expected = (arr + scalar).astype(str).tolist()

        # Act: Add the scalar to each element of the large array.
        result = basic_calculations(scalar, arr.tolist(), '+').basic_calculations()

        # Assert: Check that the result matches the expected list of string sums, validating large-scale operations.
        assert result == expected

    # Tests addition of arrays with mismatched lengths, expecting an error due to incompatible sizes.
    @pytest.mark.edge_case
    def test_addition_arrays_with_mismatched_lengths(self,basic_calculations): 
        # Arrange
        arr1 = np.random.randint(0, 1000, size=np.random.randint(400, 600)).tolist()
        arr2 = np.random.randint(0, 1000, size=np.random.randint(200, 400)).tolist()
        
       # Act & Assert: Expect an ValueError due to the bug in exception handling
        with pytest.raises(ValueError, match=(ErrorMessages.ERROR_INTERNAL_SERVER["message"])):
            basic_calculations(arr1, arr2, '+').basic_calculations()

    # Tests addition of large nested arrays (flattened with repetition), ensuring correct handling of patterned data.
    @pytest.mark.happy_path
    def test_addition_of_nested_large_arrays(self, basic_calculations):
        # Arrange: Create two 50,000-element arrays by tiling 100-element random arrays 500 times, compute expected sums.
        arr1 = np.tile(np.random.randint(-500_000, 500_000, size=100), 500)
        arr2 = np.tile(np.random.randint(-500_000, 500_000, size=100), 500)
        expected = (arr1 + arr2).astype(str).tolist()

        # Act: Perform element-wise addition of the large nested arrays.
        result = basic_calculations(arr1.tolist(), arr2.tolist(), '+').basic_calculations()

        # Assert: Verify the result matches the expected list of string sums, confirming patterned array addition.
        assert result == expected