import pytest
import numpy as np
from calculation_methods.py_calculations.basic_calculations import BasicCalculations

@pytest.fixture
def basic_calculations():
    # Fixture to provide a fresh instance of BasicCalculations for each test, ensuring consistent initialization.
    return BasicCalculations

class TestBasicCalculations:

    # Test addition of two positive scalar values, ensuring basic arithmetic functionality works as expected.
    @pytest.mark.happy_path
    def test_addition_of_two_scalars(self, basic_calculations):
        # Arrange: Generate two random positive integers (1-100) and compute their sum as a string.
        a, b = np.random.randint(1, 100), np.random.randint(1, 100)
        expected = str(a + b)
        # Act: Perform scalar addition using the Addition method.
        result = basic_calculations(a, b, '+').basic_calculations()
        # Assert: Verify the result matches the expected sum in string format, confirming correct output type.
        assert result == expected

    # Test adding a scalar to each element of an array, ensuring scalar-array operations work correctly.
    @pytest.mark.happy_path
    def test_addition_of_scalar_and_array(self, basic_calculations):
        # Arrange: Create a random scalar (1-100) and a 3-element array, compute expected sums as strings.
        scalar = np.random.randint(1, 100)
        array = np.random.randint(1, 100, size=3).tolist()
        expected = [str(scalar + x) for x in array]
        # Act: Add the scalar to the array using the Addition method.
        result = basic_calculations(scalar, array, '+').basic_calculations()
        # Assert: Check that the result is a list of strings matching the expected element-wise sums.
        assert result == expected

    # Test adding each element of an array to a scalar, verifying commutative behavior in array-scalar addition.
    @pytest.mark.happy_path
    def test_addition_of_array_and_scalar(self, basic_calculations):
        # Arrange: Generate a 3-element array and a random scalar (1-100), compute expected sums as strings.
        array = np.random.randint(1, 100, size=3).tolist()
        scalar = np.random.randint(1, 100)
        expected = [str(x + scalar) for x in array]
        # Act: Add the array to the scalar using the Addition method.
        result = basic_calculations(array, scalar, '+').basic_calculations()
        # Assert: Ensure the result matches the expected list of string sums, confirming order independence.
        assert result == expected

    # Test element-wise addition of two arrays, ensuring array-array operations produce correct sums.
    @pytest.mark.happy_path
    def test_addition_of_two_arrays(self, basic_calculations):
        # Arrange: Create two 3-element arrays of random integers (1-100), compute expected element-wise sums as strings.
        array1 = np.random.randint(1, 100, size=3).tolist()
        array2 = np.random.randint(1, 100, size=3).tolist()
        expected = [str(x + y) for x, y in zip(array1, array2)]
        # Act: Perform element-wise addition of the two arrays.
        result = basic_calculations(array1, array2, '+').basic_calculations()
        # Assert: Verify the result is a list of strings matching the expected sums.
        assert result == expected

    # Test subtraction of two scalar values, ensuring basic subtraction works and handles positive differences.
    @pytest.mark.happy_path
    def test_subtraction_of_two_scalars(self, basic_calculations):
        # Arrange: Generate two random positive integers (1-100) and compute their difference as a string.
        a, b = np.random.randint(1, 100), np.random.randint(1, 100)
        expected = str(a - b)
        # Act: Perform scalar subtraction using the Subtraction method.
        result = basic_calculations(a, b, '-').basic_calculations()
        # Assert: Check that the result matches the expected difference in string format.
        assert result == expected

    # Test subtracting an array from a scalar, ensuring each element is correctly subtracted from the scalar.
    @pytest.mark.happy_path
    def test_subtraction_of_scalar_and_array(self, basic_calculations):
        # Arrange: Create a random scalar (1-100) and a 3-element array, compute expected differences as strings.
        scalar = np.random.randint(1, 100)
        array = np.random.randint(1, 100, size=3).tolist()
        expected = [str(scalar - x) for x in array]
        # Act: Subtract the array from the scalar using the Subtraction method.
        result = basic_calculations(scalar, array, '-').basic_calculations()
        # Assert: Verify the result is a list of strings matching the expected differences.
        assert result == expected

    # Test multiplication of two scalar values, ensuring basic multiplication works correctly.
    @pytest.mark.happy_path
    def test_multiplication_of_two_scalars(self, basic_calculations):
        # Arrange: Generate two random positive integers (1-100) and compute their product as a string.
        a, b = np.random.randint(1, 100), np.random.randint(1, 100)
        expected = str(a * b)
        # Act: Perform scalar multiplication using the Multiply method.
        result = basic_calculations(a, b, '*').basic_calculations()
        # Assert: Check that the result matches the expected product in string format.
        assert result == expected

    # Test multiplying a scalar by each element of an array, ensuring scalar-array multiplication works.
    @pytest.mark.happy_path
    def test_multiplication_of_scalar_and_array(self, basic_calculations):
        # Arrange: Create a random scalar (1-100) and a 3-element array, compute expected products as strings.
        scalar = np.random.randint(1, 100)
        array = np.random.randint(1, 100, size=3).tolist()
        expected = [str(scalar * x) for x in array]
        # Act: Multiply the scalar by the array using the Multiply method.
        result = basic_calculations(scalar, array, '*').basic_calculations()
        # Assert: Verify the result is a list of strings matching the expected element-wise products.
        assert result == expected

    # Test division of two scalar values, ensuring basic division works with floating-point results.
    @pytest.mark.happy_path
    def test_division_of_two_scalars(self, basic_calculations):
        # Arrange: Generate two random positive integers (1-100) and compute their quotient as a string.
        a, b = np.random.randint(1, 100), np.random.randint(1, 100)
        expected = str(a / b)
        # Act: Perform scalar division using the Divide method.
        result = basic_calculations(a, b, '/').basic_calculations()
        # Assert: Check that the result approximates the expected quotient, accounting for float precision.
        assert result == pytest.approx(expected)

    # Test dividing each element of an array by a scalar, ensuring precision in array-scalar division.
    @pytest.mark.edge_case
    def test_division_of_array_by_scalar(self, basic_calculations):
        # Arrange: Create a 3-element array and a random scalar (1-100), compute expected quotients rounded to 10 decimals.
        array = np.random.randint(1, 100, size=3).tolist()
        scalar = np.random.randint(1, 100)
        expected = [round(x / scalar, 10) for x in array]
        # Act: Divide the array by the scalar and round the results to 10 decimals for comparison.
        result = [round(float(x), 10) for x in basic_calculations(array, scalar, '/').basic_calculations()]
        # Assert: Verify the result matches the expected list of rounded quotients, ensuring numerical accuracy.
        assert result == expected

    # Test element-wise division of two arrays, handling division by zero with 'undefined' outputs.
    @pytest.mark.edge_case
    def test_division_of_two_arrays(self, basic_calculations):
        # Arrange: Generate two 3-element arrays of random integers (1-1000), compute expected quotients or 'undefined'.
        array1 = np.random.randint(1, 1000, size=3).tolist()
        array2 = np.random.randint(1, 1000, size=3).tolist()
        expected = [str(x / y) if y != 0 else "undefined" for x, y in zip(array1, array2)]
        # Act: Perform element-wise division of the two arrays.
        result = basic_calculations(array1, array2, '/').basic_calculations()
        # Assert: Check that the result approximates the expected list, accounting for float precision and zero handling.
        assert result == expected