import pytest
import numpy as np
from calculation_methods.py_calculations.basic_calculations import BasicCalculations

@pytest.fixture
def basic_calculations():
    return BasicCalculations

class TestDivisionOperations:

    # Tests element-wise division of two arrays with non-zero elements, ensuring basic division works correctly.
    @pytest.mark.happy_path
    def test_divide_two_arrays_with_non_zero_elements(self, basic_calculations):
        # Arrange: Generate two 3-element arrays of random positive integers (1-100) and compute expected string results.
        dividend = np.random.randint(1, 100, size=3).tolist()
        divisor = np.random.randint(1, 100, size=3).tolist()
        expected = [str(x / y) for x, y in zip(dividend, divisor)]
        
        # Act: Perform element-wise division using the Divide method.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()
        
        # Assert: Verify the result approximates the expected list of string quotients, allowing for float precision.
        assert result == pytest.approx(expected)

    # Tests division when the divisor array contains a zero, expecting 'undefined' for that position.
    @pytest.mark.edge_case
    def test_divide_array_by_array_with_zero_in_divisor(self, basic_calculations):
        # Arrange: Create a dividend array and a divisor array with a zero at the end, compute expected results with 'undefined'.
        dividend = np.random.randint(1, 100, size=3).tolist()
        divisor = [np.random.randint(1, 100), np.random.randint(1, 100), 0]
        expected = [str(dividend[i] / divisor[i]) if divisor[i] != 0 else 'undefined' for i in range(3)]
        
        # Act: Perform element-wise division.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()
        
        # Assert: Ensure the result matches the expected list, including 'undefined' for division by zero.
        assert result == expected

    # Tests division when all divisor elements are zero, expecting an entirely 'undefined' result.
    @pytest.mark.edge_case
    def test_divide_array_by_array_with_all_zeros_in_divisor(self, basic_calculations):
        # Arrange: Generate a random dividend array and a divisor array of all zeros, expecting all 'undefined' outcomes.
        dividend = np.random.randint(1, 100, size=3).tolist()
        divisor = [0, 0, 0]
        expected = ['undefined', 'undefined', 'undefined']
        
        # Act: Perform element-wise division.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()
        
        # Assert: Verify the result is a list of 'undefined' values, confirming proper zero division handling.
        assert result == expected

    # Tests division of arrays with large numbers, ensuring precision and handling of big integers.
    @pytest.mark.happy_path
    def test_divide_arrays_with_large_numbers(self, basic_calculations):
        # Arrange: Create large dividend (10^9 to 10^10-1) and divisor (1 to 10^6) arrays, compute expected string results.
        dividend = np.random.randint(np.int64(1e9), np.int64(1e10) - 1, size=3, dtype=np.int64).tolist()
        divisor = np.random.randint(1, int(1e6), size=3, dtype=np.int64).tolist()  # Ensure non-zero divisor
        expected = [str(dividend[i] / divisor[i]) for i in range(3)]

        # Act: Perform element-wise division with large numbers.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()

        # Assert: Check that the result matches the expected list of string quotients after conversion.
        assert list(map(str, result)) == expected

    # Tests division with negative numbers in both arrays, verifying sign handling and zero division.
    @pytest.mark.edge_case
    def test_divide_arrays_with_negative_numbers(self, basic_calculations):
        # Arrange: Generate two arrays with random integers (-100 to 100), including negatives, compute expected results.
        dividend = np.random.randint(-100, 100, size=3).tolist()
        divisor = np.random.randint(-100, 100, size=3).tolist()
        expected = [str(x / y) if y != 0 else 'undefined' for x, y in zip(dividend, divisor)]
        
        # Act: Perform element-wise division.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()
        
        # Assert: Verify the result matches the expected list, accounting for negative quotients and zero divisors.
        assert result == expected

    # Tests dividing an array by a non-zero scalar, ensuring each element is correctly divided.
    @pytest.mark.happy_path
    def test_divide_array_by_non_zero_scalar(self, basic_calculations):
        # Arrange: Create a 3-element dividend array and a random non-zero scalar (1-100), compute expected numerical results.
        dividend = np.random.randint(1, 100, size=3).tolist()
        scalar = np.random.randint(1, 100)
        expected = [x / scalar for x in dividend]  # Keep as float instead of converting to string

        # Act: Divide the array by the scalar.
        result = basic_calculations(dividend, scalar, '/').basic_calculations()

        # Assert
        assert [float(r) for r in result] == pytest.approx(expected)

    # Tests division of two positive scalars, ensuring basic scalar division works correctly.
    @pytest.mark.happy_path
    def test_divide_two_scalars(self, basic_calculations):
        # Arrange: Generate two random positive scalars (1-100) and compute the expected quotient.
        dividend = np.random.randint(1, 100)
        divisor = np.random.randint(1, 100)
        expected = str(dividend / divisor) if dividend % divisor != 0 else str(int(dividend / divisor))

        # Act: Perform scalar division.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()

        # Assert: Verify the result matches the expected quotient as a string.
        assert result == expected

    # Tests dividing zero by a non-zero scalar, expecting a result of zero.
    @pytest.mark.edge_case
    def test_divide_zero_by_non_zero_scalar(self, basic_calculations):
        # Arrange: Set dividend to zero and use a random non-zero scalar (1-100), expecting zero as the result.
        dividend = 0
        divisor = np.random.randint(1, 100)
        expected = 0
        
        # Act: Perform scalar division.
        result = basic_calculations(dividend, divisor, '/').basic_calculations()
        
        # Assert: Ensure the result is zero, confirming zero dividend handling.
        assert result == expected

    # Tests dividing a scalar by an array with a zero, expecting 'undefined' for that position.
    @pytest.mark.edge_case
    def test_divide_scalar_by_array_with_zero(self, basic_calculations):
        # Arrange: Use a random scalar (1-100) and a 3-element divisor array with one zero, compute expected results.
        scalar = np.random.randint(1, 100)
        divisor = [np.random.randint(1, 100), np.random.randint(1, 100), 0]
        expected = [str(scalar / x) if x != 0 else 'undefined' for x in divisor]
        
        # Act: Divide the scalar by the array.
        result = basic_calculations(scalar, divisor, '/').basic_calculations()
        
        # Assert: Verify the result approximates the expected list, with 'undefined' for division by zero.
        assert result == pytest.approx(expected)