import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.basic_calculations import BasicCalculations

@pytest.fixture
def basic_calculations():
    return BasicCalculations

class TestBasicCalculationsMultiplication:

    # Tests element-wise multiplication of two arrays with positive integers, ensuring basic array multiplication works.
    @pytest.mark.happy_path
    def test_multiply_arrays_positive_integers(self, basic_calculations):
        # Arrange: Generate two arrays of 3 random positive integers (1-100) and compute expected string results.
        array1 = np.random.randint(1, 100, size=3).tolist()
        array2 = np.random.randint(1, 100, size=3).tolist()
        expected = [str(array1[i] * array2[i]) for i in range(3)]
        
        # Act: Perform element-wise multiplication using the Multiply method.
        result = basic_calculations(array1, array2, '*').basic_calculations()
        
        # Assert: Verify the result matches the expected list of string products.
        assert result == expected

    # Tests multiplication of arrays where some elements are zero, ensuring zero is handled correctly.
    @pytest.mark.edge_case
    def test_multiply_arrays_with_zero(self, basic_calculations):
        # Arrange: Create two arrays with zeros at different positions and compute expected string results.
        array1 = [0, np.random.randint(1, 100), np.random.randint(1, 100)]
        array2 = [np.random.randint(1, 100), 0, np.random.randint(1, 100)]
        expected = [str(array1[i] * array2[i]) for i in range(3)]

        # Act: Perform element-wise multiplication.
        result = basic_calculations(array1, array2, '*').basic_calculations()

        # Assert: Ensure the result matches the expected list, converting result to strings for comparison.
        assert list(map(str, result)) == expected

    # Tests multiplication of arrays with negative numbers, verifying correct sign handling in products.
    @pytest.mark.edge_case
    def test_multiply_arrays_with_negative_numbers(self, basic_calculations):
        # Arrange: Generate two arrays with random integers (-100 to 100), including negatives, and compute expected results.
        array1 = np.random.randint(-100, 100, size=3).tolist()
        array2 = np.random.randint(-100, 100, size=3).tolist()
        expected = [str(array1[i] * array2[i]) for i in range(3)]
        
        # Act: Perform element-wise multiplication.
        result = basic_calculations(array1, array2, '*').basic_calculations()
        
        # Assert: Check that the result correctly reflects multiplication with negative numbers as strings.
        assert result == expected

    # Tests multiplication of an empty array with a non-empty array, ensuring empty input handling returns an empty result.
    @pytest.mark.edge_case
    def test_multiply_array_with_empty_array(self, basic_calculations):
        # Arrange: Use an empty array and a random 3-element array, expecting an empty result.
        array1 = []
        array2 = np.random.randint(1, 100, size=3).tolist()
        
        # Act & Assert: Expect an ValueError due to the bug in exception handling
        with pytest.raises(ValueError, match=(ErrorMessages.ERROR_INTERNAL_SERVER["message"])):
            basic_calculations(array1, array2, '-').basic_calculations()

    # Tests multiplication of arrays with large numbers, checking for overflow or precision issues.
    @pytest.mark.edge_case
    def test_multiply_arrays_with_large_numbers(self, basic_calculations):
        # Arrange: Generate two arrays with large integers (10^4 to 10^6) and compute expected string results.
        array1 = np.random.randint(1e4, 1e6, size=3).tolist()  # Corrected range
        array2 = np.random.randint(1e4, 1e6, size=3).tolist()
        expected = [str(array1[i] * array2[i]) for i in range(3)]

        # Act: Perform element-wise multiplication with large numbers.
        result = basic_calculations(array1, array2, '*').basic_calculations()

        # Assert: Ensure the result matches the expected list, converting to strings for consistency.
        assert list(map(str, result)) == expected

    # Tests multiplying an array by a scalar, ensuring each element is correctly scaled.
    @pytest.mark.happy_path
    def test_multiply_array_by_scalar(self, basic_calculations):
        # Arrange: Create a 3-element array and a random scalar (1-100), compute expected string results.
        array = np.random.randint(1, 100, size=3).tolist()
        scalar = np.random.randint(1, 100)
        expected = [str(array[i] * scalar) for i in range(3)]
        
        # Act: Multiply the array by the scalar.
        result = basic_calculations(array, scalar, '*').basic_calculations()
        
        # Assert: Verify the result matches the expected list of scaled values as strings.
        assert result == expected

    # Tests multiplying an array by zero, ensuring all elements result in zero.
    @pytest.mark.edge_case
    def test_multiply_array_with_zero_scalar(self, basic_calculations):
        # Arrange: Use a random 3-element array and a scalar of 0, expecting all zeros.
        array = np.random.randint(1, 100, size=3).tolist()
        scalar = 0
        expected = [0, 0, 0]
        
        # Act: Multiply the array by zero.
        result = basic_calculations(array, scalar, '*').basic_calculations()
        
        # Assert: Check that the result is a list of zeros, confirming zero multiplication behavior.
        assert result == expected

    # Tests multiplying a negative array by a positive scalar, verifying sign preservation.
    @pytest.mark.edge_case
    def test_multiply_negative_array_by_scalar(self, basic_calculations):
        # Arrange: Create a 3-element array of negative numbers (-100 to -1) and a positive scalar (1-100).
        array = np.random.randint(-100, -1, size=3).tolist()
        scalar = np.random.randint(1, 100)
        expected = [str(array[i] * scalar) for i in range(3)]
        
        # Act: Multiply the negative array by the scalar.
        result = basic_calculations(array, scalar, '*').basic_calculations()
        
        # Assert: Ensure the result correctly reflects negative products as strings.
        assert result == expected

    # Tests multiplying a large array by a scalar, checking for precision and overflow handling.
    @pytest.mark.edge_case
    def test_multiply_large_scalar_and_array(self, basic_calculations):
        # Arrange: Generate a 3-element array of large numbers (10^6 to 10^7) and a small scalar (1-10).
        array = np.random.randint(1e6, 1e7, size=3).tolist()
        scalar = np.random.randint(1, 10)
        expected = [str(array[i] * scalar) for i in range(3)]
        
        # Act: Multiply the large array by the scalar.
        result = basic_calculations(array, scalar, '*').basic_calculations()
        
        # Assert: Verify the result matches the expected scaled values as strings.
        assert result == expected

    # Tests multiplication of two positive scalars, ensuring basic scalar multiplication works.
    @pytest.mark.happy_path
    def test_multiply_scalars(self, basic_calculations):
        # Arrange: Use two random positive scalars (1-100) and compute the expected string product.
        scalar1 = np.random.randint(1, 100)
        scalar2 = np.random.randint(1, 100)
        expected = str(scalar1 * scalar2)
        
        # Act: Perform scalar multiplication.
        result = basic_calculations(scalar1, scalar2, '*').basic_calculations()
        
        # Assert: Check that the result matches the expected product as a string.
        assert result == expected

    # Tests multiplication of a positive scalar with a negative scalar, verifying sign handling.
    @pytest.mark.edge_case
    def test_multiply_scalar_and_negative_scalar(self, basic_calculations):
        # Arrange: Use a positive scalar (1-100) and a negative scalar (-100 to -1), compute the expected result.
        scalar1 = np.random.randint(1, 100)
        scalar2 = np.random.randint(-100, -1)
        expected = str(scalar1 * scalar2)
        
        # Act: Perform multiplication with mixed signs.
        result = basic_calculations(scalar1, scalar2, '*').basic_calculations()
        
        # Assert: Ensure the result correctly reflects the negative product as a string.
        assert result == expected

    # Tests multiplication of two large scalars, checking for precision and overflow issues.
    @pytest.mark.edge_case
    def test_multiply_large_scalars(self, basic_calculations):
        # Arrange: Generate two large scalars (10^6 to 10^7) and compute the expected string product.
        scalar1 = np.random.randint(1e6, 1e7)
        scalar2 = np.random.randint(1e6, 1e7)
        expected = str(scalar1 * scalar2)
        
        # Act: Perform multiplication with large scalars.
        result = basic_calculations(scalar1, scalar2, '*').basic_calculations()
        
        # Assert: Verify the result matches the expected large product as a string.
        assert result == expected

    # Tests multiplication of a float and an integer, ensuring mixed-type handling and precision.
    @pytest.mark.edge_case
    def test_multiply_float_and_integer(self, basic_calculations):
        # Arrange: Use a random float (1-100, 2 decimal places) and an integer (1-100), compute expected result.
        scalar1 = round(np.random.uniform(1, 100), 2)
        scalar2 = np.random.randint(1, 100)
        expected = str(scalar1 * scalar2)
        
        # Act: Perform multiplication with mixed types.
        result = basic_calculations(scalar1, scalar2, '*').basic_calculations()
        
        # Assert: Check that the result approximates the expected product, accounting for float precision.
        assert result == pytest.approx(expected)