import random
import pytest
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.basic_calculations import BasicCalculations
import numpy as np

@pytest.fixture
def basic_calculations():
    return lambda x, y, operator: BasicCalculations(x, y, operator)

class TestSubtraction:
    # Tests subtraction of two positive scalars where the first is larger, ensuring basic functionality works.
    @pytest.mark.happy_path
    def test_subtraction_scalar_scalar_happy_path(self, basic_calculations):
        # Arrange: Choose a larger random number (10-100) and a smaller one (1-9) to ensure a positive result.
        scalar1 = random.randint(10, 100)
        scalar2 = random.randint(1, 9)

        # Act: Perform subtraction using the BasicCalculations class.
        result = basic_calculations(scalar1, scalar2, '-').basic_calculations()

        # Assert: Verify the result matches the expected subtraction, converted to string as per assumed class behavior.
        assert result == str(scalar1 - scalar2)

    # Tests subtraction resulting in a negative number, ensuring the method handles negative outcomes correctly.
    @pytest.mark.happy_path
    def test_subtraction_scalar_scalar_negative_result(self, basic_calculations):
        # Arrange: Select a smaller number (1-50) and a larger one (51-100) to guarantee a negative result.
        scalar1 = random.randint(1, 50)
        scalar2 = random.randint(51, 100)

        # Act: Perform subtraction.
        result = basic_calculations(scalar1, scalar2, '-').basic_calculations()

        # Assert: Check that the result correctly reflects the negative difference as a string.
        assert result == str(scalar1 - scalar2)

    # Tests subtraction where the result is zero, an edge case to ensure proper handling of zero outcomes.
    @pytest.mark.edge_case
    def test_subtraction_scalar_scalar_zero_result(self, basic_calculations):
        # Arrange: Use the same random number for both inputs to ensure the result is zero.
        scalar1 = random.randint(1, 100)
        scalar2 = scalar1

        # Act: Perform subtraction.
        result = basic_calculations(scalar1, scalar2, '-').basic_calculations()

        # Assert: Expect the result to be 0 (assuming the method might return an integer here instead of a string).
        assert result == 0

    # Tests subtraction involving zero as the first operand, ensuring the method handles zero correctly.
    @pytest.mark.edge_case
    def test_subtraction_scalar_scalar_with_zero(self, basic_calculations):
        # Arrange: Use 0 as the first number and a random positive number as the second.
        scalar2 = random.randint(1, 100)  

        # Act: Perform subtraction (0 - scalar2).
        result = basic_calculations(0, scalar2, '-').basic_calculations()

        # Assert: Verify the result is the negative of scalar2, formatted as a string.
        assert result == str(0 - scalar2)  

    # Tests subtraction with very large numbers to ensure the method handles big integers without overflow or precision issues.
    @pytest.mark.edge_case
    def test_subtraction_scalar_scalar_large_numbers(self, basic_calculations):
        # Arrange: Generate a large number (10^10 to 10^12) and a smaller subtractor (1 to 10^6).
        large_number = random.randint(10**10, 10**12)
        subtractor = random.randint(1, 10**6)
        expected = str(large_number - subtractor)
        
        # Act: Perform subtraction with large numbers.
        result = basic_calculations(large_number, subtractor, '-').basic_calculations()
        # Assert: Ensure the result matches the expected string representation.
        assert result == expected

    # Tests subtraction with very small floating-point numbers to check precision with tiny values.
    @pytest.mark.edge_case
    def test_subtraction_scalar_scalar_small_numbers(self, basic_calculations):
        # Arrange: Use small random floats (between 1e-3 and 1e-4, and 1e-3 and 1e-2) to test precision.
        small_number = random.uniform(1e-3, 1e-4)
        subtractor = random.uniform(1e-3, 1e-2)

        # Act: Perform subtraction with small numbers.
        result = basic_calculations(small_number, subtractor, '-').basic_calculations()

        # Assert: Verify the result matches the expected difference as a string.
        assert result == str(small_number - subtractor)

    # Tests subtraction of a scalar from each element in a list, ensuring list handling works correctly.
    @pytest.mark.happy_path
    def test_subtraction_scalar_list(self, basic_calculations):
        # Arrange: Create a scalar and a list of 5 random numbers (1-20), compute expected results.
        scalar = random.randint(1, 50)
        numbers = [random.randint(1, 20) for _ in range(5)]
        expected = ([scalar - num for num in numbers])
        # Act: Subtract each list element from the scalar.
        result = basic_calculations(scalar, numbers, '-').basic_calculations()

        # Assert: Convert result to integers and compare with expected list.
        assert list(map(int, result)) == expected

    # Tests subtraction of a scalar from each element in a list (reversed order), verifying commutative handling.
    @pytest.mark.edge_case
    def test_subtraction_list_scalar(self, basic_calculations):
        # Arrange: Create a list of 5 random numbers and a scalar, compute expected results.
        numbers = [random.randint(1, 50) for _ in range(5)]
        scalar = random.randint(1, 50)
        expected = [num - scalar for num in numbers]
        
        # Act: Subtract the scalar from each list element.
        result = basic_calculations(numbers, scalar, '-').basic_calculations()

        # Assert: Ensure the result matches the expected list after integer conversion.
        assert list(map(int, result)) == expected

    # Tests element-wise subtraction of two lists of the same length, ensuring proper list-to-list operation.
    @pytest.mark.edge_case
    def test_subtraction_list_list_same_length(self, basic_calculations):
        # Arrange: Create two lists of 5 random numbers each, compute expected element-wise differences.
        list1 = [random.randint(1, 100) for _ in range(5)]
        list2 = [random.randint(1, 100) for _ in range(5)]
        expected = [a - b for a, b in zip(list1, list2)]
        
        # Act: Perform element-wise subtraction.
        result = basic_calculations(list1, list2, '-').basic_calculations()

        # Assert: Verify the result matches the expected list after integer conversion.
        assert list(map(int, result)) == expected

    # Tests subtraction with lists of different lengths, expecting an error due to mismatched sizes.
    @pytest.mark.edge_case
    def test_subtraction_list_list_different_length(self, basic_calculations):
        # Arrange: Create two lists with different lengths (7 and 5).
        list1 = [random.randint(1, 100) for _ in range(7)]
        list2 = [random.randint(1, 100) for _ in range(5)]

        # Act & Assert: Expect an ValueError due to the bug in exception handling
        with pytest.raises(ValueError, match=(ErrorMessages.ERROR_INTERNAL_SERVER["message"])):
            basic_calculations(list1, list2, '-').basic_calculations()

    # Tests subtraction with an empty list, ensuring the method handles empty inputs gracefully.
    @pytest.mark.edge_case
    def test_subtraction_empty_list(self, basic_calculations):
        # Arrange: Use an empty list and a random scalar.
        empty_list = []
        scalar = random.randint(1, 100)

        # Act: Perform subtraction with an empty list.
        result = basic_calculations(empty_list, scalar, '-').basic_calculations()

        # Assert: Expect an empty list as the result, indicating no operation was performed.
        assert result == []

    # Tests subtraction with floating-point numbers, ensuring precision and float handling.
    @pytest.mark.happy_path
    def test_subtraction_floats(self, basic_calculations):
        # Arrange: Generate two random floats for subtraction.
        scalar1 = random.uniform(1.1, 50.5)
        scalar2 = random.uniform(0.1, 10.1)

        # Act: Perform subtraction with floats.
        result = basic_calculations(scalar1, scalar2, '-').basic_calculations()

        # Assert: Compare the float result with the expected difference, allowing for tiny precision errors.
        assert float(result) == pytest.approx(scalar1 - scalar2, rel=1e-9)

    # Tests subtraction with mixed integer and float inputs, ensuring type compatibility.
    @pytest.mark.edge_case
    def test_subtraction_mixed_integer_float(self, basic_calculations):
        # Arrange: Use a random integer and a random float.
        scalar1 = random.randint(1, 50)
        scalar2 = random.uniform(0.1, 10.1)

        # Act: Perform subtraction with mixed types.
        result = basic_calculations(scalar1, scalar2, '-').basic_calculations()

        # Assert: Verify the result matches the expected difference within floating-point precision.
        assert float(result) == pytest.approx(scalar1 - scalar2, rel=1e-9)

    # Tests subtraction with an invalid input type (string), expecting an error for type mismatch.
    @pytest.mark.error_case
    def test_subtraction_invalid_type(self, basic_calculations):
        # Arrange: Use a string as the first input and an integer as the second.
        scalar1 = "abc"
        scalar2 = 5

        # Act & Assert: Expect a ValueError due to incompatible input type.
        with pytest.raises(ValueError, match=(ErrorMessages.ERROR_INVALID_DATA["message"])):
            basic_calculations(scalar1, scalar2, '-').basic_calculations()