import re
from django.test import Client
import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.blank_corrections import BlankCorrections
from calculation.calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from unittest.mock import patch

class TestBlankCorrection:
    client = Client()

    # Tests blank correction with typical arrays of blank values and input data, ensuring correct subtraction of the mean.
    @pytest.mark.happy_path
    def test_blank_correction_typical_case(self):
        # Arrange:
        blank_values = np.random.uniform(1.0, 5.0, size=3).tolist()
        input_data = np.random.uniform(5.0, 10.0, size=3).tolist()
        bc = BlankCorrections(blank_values, input_data)
        expected_output = [x - np.mean(blank_values) for x in input_data]
        
        # Act: Apply blank correction by subtracting the mean of blank values from each input datum.
        result = bc.blank_correction()
        
        # Assert: Verify the result matches the expected list of corrected values, confirming typical case functionality.
        assert result == expected_output
    
    # Tests blank correction with a single blank value, ensuring the method works with minimal blank data.
    @pytest.mark.happy_path
    def test_blank_correction_single_value(self):
        # Arrange: 
        blank_values = [np.random.uniform(1.0, 5.0)]
        input_data = np.random.uniform(5.0, 10.0, size=3).tolist()
        bc = BlankCorrections(blank_values, input_data)
        expected_output = [x - np.mean(blank_values) for x in input_data]
        
        # Act: Apply blank correction using the single blank value's mean.
        result = bc.blank_correction()
        
        # Assert: Check that the result matches the expected list, confirming functionality with a single blank value.
        assert result == expected_output
    
    # Tests blank correction with empty input data, expecting an empty result as no correction is needed.
    @pytest.mark.edge_case
    def test_blank_correction_empty_input_data(self):
        # Arrange: 
        blank_values = np.random.uniform(1.0, 5.0, size=3).tolist()
        input_data = []
        bc = BlankCorrections(blank_values, input_data)
        expected_output = []
        
        # Act: Apply blank correction to the empty input data.
        result = bc.blank_correction()
        
        # Assert: Verify the result is an empty list, ensuring proper handling of no input data.
        assert result == expected_output
    
    # Tests blank correction with empty blank values, expecting a ValueError due to invalid blank data.
    @pytest.mark.edge_case
    def test_blank_values_empty(self):
        # Arrange:
        blank_values = []
        input_data = np.random.uniform(1.0, 10.0, size=3).tolist()
        bc = BlankCorrections(blank_values, input_data)

        # Extract the expected error message string from the wrapped exception
        original_message = ErrorMessages.ERROR_BLANK_VALUES_EMPTY["message"]
        expected_code = ErrorMessages.ERROR_BLANK_CORRECTION_FAILED["code"]  # 'BlankCorrectionFailed'
        expected_message = ErrorMessages.ERROR_BLANK_CORRECTION_FAILED["message"].format(
            str((ErrorMessages.ERROR_BLANK_VALUES_EMPTY["code"], original_message))
        )

        # Act & Assert: Expect a ValueError with the correct message
        with pytest.raises(ValueError, match=re.escape(expected_message)) as exc_info:
            bc.blank_correction()

        # Verify error code and message
        assert exc_info.value.args[0] == expected_code  # 'BlankCorrectionFailed'
        assert exc_info.value.args[1] == expected_message
    # Tests initialization with valid blank values and input data, ensuring attributes are set correctly.
    @pytest.mark.happy_path
    def test_init_with_valid_blank_values_and_input_data(self):
        # Arrange:
        blank_values = np.random.uniform(1.0, 5.0, size=3).tolist()
        input_data = np.random.uniform(5.0, 10.0, size=3).tolist()
        
        # Act: Initialize the BlankCorrections object with the arranged data.
        bc = BlankCorrections(blank_values=blank_values, input_data=input_data)
        
        # Assert: Check that the object's attributes match the input, confirming proper initialization.
        assert bc.blank_values == blank_values
        assert bc.input_data == input_data
    
    # Tests initialization with no arguments (None values), ensuring default attributes are set to None.
    @pytest.mark.happy_path
    def test_init_with_none_values(self):
        # Act: Initialize the BlankCorrections object without providing blank_values or input_data.
        bc = BlankCorrections()
        
        # Assert: Verify that both attributes are None, confirming default initialization behavior.
        assert bc.blank_values is None
        assert bc.input_data is None
    
    # Tests initialization with empty blank values, ensuring the object handles an empty list correctly.
    @pytest.mark.edge_case
    def test_init_with_empty_blank_values(self):
        # Arrange:
        blank_values = []
        input_data = np.random.uniform(5.0, 10.0, size=3).tolist()
        
        # Act: Initialize the BlankCorrections object with empty blank values.
        bc = BlankCorrections(blank_values=blank_values, input_data=input_data)
        
        # Assert: Check that the attributes match the input, confirming proper handling of empty blank values.
        assert bc.blank_values == blank_values
        assert bc.input_data == input_data
    
    # Tests initialization with a non-list (scalar) blank value, ensuring flexibility in input type handling.
    @pytest.mark.edge_case
    def test_init_with_non_list_blank_values(self):
        # Arrange: 
        blank_values = np.random.uniform(1.0, 5.0)
        input_data = np.random.uniform(5.0, 10.0, size=3).tolist()
        
        # Act: Initialize the BlankCorrections object with a scalar blank value.
        bc = BlankCorrections(blank_values=blank_values, input_data=input_data)
        
        # Assert: Verify the attributes match the input, confirming the object accepts non-list blank values.
        assert bc.blank_values == blank_values
        assert bc.input_data == input_data
    
    # Tests initialization with a non-list (scalar) input data value, ensuring flexibility in input type handling.
    @pytest.mark.edge_case
    def test_init_with_non_list_input_data(self):
        # Arrange: 
        blank_values = np.random.uniform(1.0, 5.0, size=3).tolist()
        input_data = np.random.uniform(5.0, 10.0)
        
        # Act: Initialize the BlankCorrections object with a scalar input data value.
        bc = BlankCorrections(blank_values=blank_values, input_data=input_data)
        
        # Assert: Check that the attributes match the input, confirming the object accepts non-list input data.
        assert bc.blank_values == blank_values
        assert bc.input_data == input_data