import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.cubic_spline import CubicSplineInterpolator
from calculation.py_tests.unit_test_utils import Utils  
from unittest.mock import patch

# Mocking the compute_statistical_cv function
def mock_compute_statistical_cv(concentrations):
    return np.std(concentrations) / np.mean(concentrations) * 100 if concentrations else None

@pytest.fixture
def sample_data():
    # Use the Utils class static method to generate sample data.
    return Utils.generate_sample_data()

class TestCalculateCV:
    # Tests calculate_cv with normal input data (S4), ensuring CV is calculated correctly for a standard.
    @pytest.mark.happy_path
    def test_calculate_cv_normal_case(self, sample_data):
        # Arrange: Use the sample data from Utils.
        input_data = sample_data["input_data"]
        standard_values = sample_data["standard_values"]
        blank_values = sample_data["blank_values"]
        
        # Act: Initialize interpolator and calculate CV for S4.
        interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values)
        
        # Assert: Patch compute_statistical_cv from calculation_utils and verify CV is a valid float.
        with patch('calculation_methods.py_calculations.calculation_utils.compute_statistical_cv', side_effect=mock_compute_statistical_cv):
            cv = interpolator.calculate_cv('S4')
            assert cv is not None
            assert isinstance(cv, float)
            
    # Tests calculate_cv with a different standard (S1), ensuring it handles various identifiers.
    @pytest.mark.happy_path
    def test_calculate_cv_different_sample_type(self, sample_data):
        # Arrange: Use the sample data from Utils.
        input_data = sample_data["input_data"]
        standard_values = sample_data["standard_values"]
        blank_values = sample_data["blank_values"]
        
        # Act: Initialize interpolator and calculate CV for S1.
        interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values)
        
        # Assert: Patch compute_statistical_cv from calculation_utils and verify CV is a valid float.
        with patch('calculation_methods.py_calculations.calculation_utils.compute_statistical_cv', side_effect=mock_compute_statistical_cv):
            cv = interpolator.calculate_cv('S1')
            assert cv is not None
            assert isinstance(cv, float)

    # Tests calculate_cv when division by zero occurs, ensuring it returns None gracefully.
    @pytest.mark.edge_case
    def test_calculate_cv_division_by_zero(self, sample_data):
        # Arrange: Use the sample data from Utils.
        input_data = sample_data["input_data"]
        standard_values = sample_data["standard_values"]
        blank_values = sample_data["blank_values"]
        
        # Act: Initialize interpolator and calculate CV for S3 with mocked zero-division.
        interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values)
        
        # Assert: Patch compute_statistical_cv from calculation_utils to return None, verify CV is None.
        with patch('calculation_methods.py_calculations.calculation_utils.compute_statistical_cv', return_value=None):
            cv = interpolator.calculate_cv('S3')
            assert cv is None