import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.cubic_spline import CubicSplineInterpolator
from calculation.calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation.py_tests.unit_test_utils import Utils  

@pytest.fixture
def sample_data():
    # Use Utils.generate_sample_data() to generate the sample data dynamically.
    return Utils.generate_sample_data()

class TestExtractCoefficients:

    @pytest.mark.happy_path
    def test_extract_coefficients_normal_case(self, sample_data):
        # Test extract_coefficients under normal conditions with valid input data from Utils.
        # Arrange
        input_data = sample_data['input_data']
        standard_values = sample_data['standard_values']
        blank_values = sample_data['blank_values']
        
        # Act
        interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values)
        coefficients = interpolator.extract_coefficients()
        
        # Assert
        assert isinstance(coefficients, list)
        assert len(coefficients) > 0
        for coeff in coefficients:
            assert 'identifier' in coeff
            assert 'x_start' in coeff
            assert 'x_end' in coeff
            assert 'a' in coeff
            assert 'b' in coeff
            assert 'c' in coeff
            assert 'd' in coeff

    @pytest.mark.happy_path
    def test_extract_coefficients_weighted(self, sample_data):
        # Test extract_coefficients with weighted option enabled using data from Utils.
        # Arrange
        input_data = sample_data['input_data']
        standard_values = sample_data['standard_values']
        blank_values = sample_data['blank_values']
        
        # Act and Assert
        interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values, weighted=True)
        coefficients = interpolator.extract_coefficients()
        # Note: Added basic assertions for consistency; adjust based on expected weighted behavior if needed.
        assert isinstance(coefficients, list)
        assert len(coefficients) > 0

    @pytest.mark.edge_case
    def test_extract_coefficients_empty_data(self):
        # Test extract_coefficients with empty input data to ensure proper error handling.
        # Arrange
        input_data = []
        standard_values = []
        blank_values = []
        
        expected_error_message = ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]  
        
        # Act & Assert
        with pytest.raises(ValueError, match=expected_error_message):
            CubicSplineInterpolator(input_data, standard_values, blank_values)

    @pytest.mark.edge_case
    def test_extract_coefficients_identical_x_values(self):
        # Test extract_coefficients with identical x values in input data to check error handling.
        # Arrange
        input_data = [
            {'identifier': 'S1', 'x': [1], 'y': [2, 3, 4]}
        ]
        standard_values = [
            {'identifier': 'S1', 'x': [1], 'y': [2, 3, 4]}
        ]
        blank_values = [
            {'identifier': 'S1', 'x': [1], 'y': [0, 0, 0]}
        ]

        expected_error_message = ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]  

        # Act & Assert
        with pytest.raises(ValueError, match=expected_error_message):
            CubicSplineInterpolator(input_data, standard_values, blank_values)