import re
import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.calculation_constants.constants import CUBICMODEL, FIVEPLMODEL, FOURPLMODEL, LINEARMODEL
from calculation_methods.py_calculations.calculation_utils import generate_interpolated_curve_data
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

class TestGenerateInterpolatedCurveData:
    #Tests for generate_interpolated_curve_data function.

    @staticmethod
    def generate_random_test_data(curve_type):
        #Helper function to generate random test data.
        np.random.seed(42)  # For reproducibility
        x_data = np.sort(np.random.uniform(1, 100, 10))  # Random x-values
        y_data = np.random.uniform(10, 500, 10)  # Random y-values

        # Generate coefficients based on curve type
        if curve_type == LINEARMODEL:
            coefficients = {"a": np.random.uniform(1, 10), "b": np.random.uniform(1, 10)}
        elif curve_type in [FOURPLMODEL, FIVEPLMODEL]:
            coefficients = {"a": 0, "b": 1, "c": np.median(x_data), "d": max(y_data), "g": 1}
        elif curve_type == CUBICMODEL:
            coefficients = {"a": np.random.uniform(-2, 2), "b": np.random.uniform(-2, 2),
                            "c": np.random.uniform(-2, 2), "d": np.random.uniform(-2, 2)}
        else:
            coefficients = None

        return x_data.tolist(), y_data.tolist(), coefficients

    
    def test_generate_interpolated_curve_cubic(self):
        #Test cubic spline interpolation.
        # Arrange
        x_data, y_data, coefficients = self.generate_random_test_data(CUBICMODEL)

        # Act
        result = generate_interpolated_curve_data(x_data= x_data, y_data=y_data, curve_type =CUBICMODEL, coefficients = coefficients)

        # Assert
        assert isinstance(result, list)
        assert len(result) >= 20
        assert all("x_coordinate" in pt and "y_coordinate" in pt for pt in result)

    def test_generate_interpolated_curve_4pl(self):
        #Test 4PL curve interpolation.
        # Arrange
        x_data, y_data, coefficients = self.generate_random_test_data(FOURPLMODEL)

        # Act
        result = generate_interpolated_curve_data(x_data=x_data, y_data =y_data, curve_type = FOURPLMODEL, coefficients =coefficients)

        # Assert
        assert isinstance(result, list)
        assert len(result) >= 20
        assert all("x_coordinate" in pt and "y_coordinate" in pt for pt in result)

    def test_generate_interpolated_curve_5pl(self):
        #Test 5PL curve interpolation.
        # Arrange
        x_data, y_data, coefficients = self.generate_random_test_data(FIVEPLMODEL)

        # Act
        result = generate_interpolated_curve_data(x_data=x_data, y_data =y_data, curve_type = FIVEPLMODEL, coefficients =coefficients)

        # Assert
        assert isinstance(result, list)
        assert len(result) >= 20
        assert all("x_coordinate" in pt and "y_coordinate" in pt for pt in result)

    @pytest.mark.edge_case
    def test_generate_interpolated_curve_invalid_type(self):
        # Test error handling for invalid curve type.
        # Arrange
        x_data, y_data, coefficients = self.generate_random_test_data(LINEARMODEL)
        
        # Expected error details
        expected_message = ErrorMessages.ERROR_GENERATE_CURVE_POINTS["message"]
        with pytest.raises(ValueError, match=(expected_message)):
            generate_interpolated_curve_data(x_data=x_data, y_data=y_data, curve_type='invalidcurve', coefficients=coefficients)
        assert expected_message
