import logging
import os
import shutil
import unittest
from datetime import datetime, timedelta
from glob import glob
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.delete_logfiles import LogCleanupMiddleware, create_log_directory
from calculation_methods.py_calculations.calculation_constants.constants import LOG_DIR

class TestLogCleanupMiddleware(unittest.TestCase):

    def setUp(self):
        #Arrange: Create sample log files for testing.
        os.makedirs(LOG_DIR, exist_ok=True)

        # Generate test log files
        self.today_log = os.path.join(LOG_DIR, f"debug_info_{datetime.now().strftime('%Y-%m-%d')}.log")
        self.old_log_1 = os.path.join(LOG_DIR, f"debug_info_{(datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')}.log")
        self.old_log_2 = os.path.join(LOG_DIR, f"debug_info_{(datetime.now() - timedelta(days=2)).strftime('%Y-%m-%d')}.log")
        self.non_debug_log = os.path.join(LOG_DIR, "random_log.log")  # Should not be deleted

        # Create test log files
        for log_file in [self.today_log, self.old_log_1, self.old_log_2, self.non_debug_log]:
            with open(log_file, "w") as f:
                f.write("Test log data\n")

    def tearDown(self):
        #Clean up log files after tests.
        for log_file in glob(os.path.join(LOG_DIR, "*.log")):
            os.remove(log_file)

    def test_create_log_directory(self):
        #Verify that the log directory is created if it doesn't exist.
        # Arrange: Remove the directory if it exists.
        if os.path.exists(LOG_DIR):
            shutil.rmtree(LOG_DIR)  # Remove the entire directory and its contents.

        # Act
        create_log_directory()  # Directly call the standalone function

        # Assert
        self.assertTrue(os.path.exists(LOG_DIR), "Log directory should be created.")
    
    def test_todays_log_created_by_logging(self):
        #Verify that today's log file is created when logging is used."""

        # Arrange: Setup expected log file path
        log_dir = create_log_directory()
        today_log = os.path.join(log_dir, f"debug_info_{datetime.now().strftime('%Y-%m-%d')}.log")

        # Ensure the log file does not exist before logging
        if os.path.exists(today_log):
            os.remove(today_log)

        # Act: Explicitly configure logging in the test
        logging.basicConfig(
            level=logging.DEBUG,
            handlers=[
                logging.FileHandler(today_log, mode='w')
            ],
            format="%(levelname)s %(asctime)s %(message)s"
        )

        logger = logging.getLogger("log_info")
        logger.setLevel(logging.INFO)  # Ensure correct logging level
        logger.info("Test log entry to trigger file creation.")  # Write log message

        # Assert: Check if the log file is created
        self.assertTrue(os.path.exists(today_log))


    def test_delete_old_logs(self):
        #Verify that only old debug logs are deleted, and today's log remains.
        # Arrange
        middleware = LogCleanupMiddleware(lambda request: None)  # Fix: Provide a dummy response function

        # Act
        logs_deleted = middleware.delete_old_logs()

        # Assert
        self.assertTrue(logs_deleted, "Previuos debug_info logs deleted")
        
    def test_no_old_logs_to_delete(self):
        #Ensure no errors occur when no old logs exist.
        # Arrange
        for log_file in [self.old_log_1, self.old_log_2]:
            os.remove(log_file)

        middleware = LogCleanupMiddleware(lambda request: None) 

        # Act
        logs_deleted = middleware.delete_old_logs()

        # Assert
        self.assertFalse(logs_deleted, "No logs should be deleted if none exist")
    
    def test_delete_old_logs_raises_error_when_file_is_opened(self):
        # Test that delete_old_logs raises an error when a log file cannot be deleted because it's in use.
        # Arrange
        middleware = LogCleanupMiddleware(lambda request: None)
        log_file = os.path.join(LOG_DIR, f"debug_info_{(datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')}.log")

        os.makedirs(LOG_DIR, exist_ok=True)  # Ensure log directory exists

        # Create log file and keep it open
        with open(log_file, "w") as f:
            f.write("Test data")

            # Act & Assert: Expect `handle_error()` to raise a ValueError
            with self.assertRaises(ValueError) as context:
                middleware.delete_old_logs()

        # Verify the error code and message from the ValueError
        self.assertEqual(context.exception.args[0], ErrorMessages.ERROR_INTERNAL_SERVER ["code"])  # 'DeleteLogfiles'
        self.assertEqual(context.exception.args[1], ErrorMessages.ERROR_INTERNAL_SERVER ["message"])  # 'Error deleting log file.'

        # Cleanup: Ensure the file is deleted after test execution
        os.remove(log_file)



