import pytest
from calculation_methods.py_calculations.cubic_spline import CubicSplineInterpolator
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error

# Mock handle_error to capture errors during testing
def mock_handle_error(error_obj):
    raise ValueError(error_obj["message"])

# Patch the handle_error function for the duration of the tests
@pytest.fixture(autouse=True)
def patch_handle_error(monkeypatch):
    monkeypatch.setattr('calculation_methods.py_calculations.calculation_utils.handle_error', mock_handle_error)

# Test case for input with less than three data points, expecting ValueError due to insufficient data
def test_insufficient_data_points_less_than_three():
    # Arrange: Input with less than 3 data points
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [0.5], 'coordinates': 'A1'},
        {'identifier': 'S2', 'x': [2], 'y': [0.7], 'coordinates': 'A2'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to insufficient data points
    with pytest.raises(ValueError) as exc_info:
        CubicSplineInterpolator(input_data, standard_values, blank_values)
    
    # Extract the actual error message
    actual_error_message = str(exc_info.value)

    # Verify the error message matches expected text
    assert actual_error_message == ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]

# Test case for completely empty input data, expecting ValueError due to no data points
def test_no_data_points():
    # Arrange: Completely empty input data
    input_data = []
    standard_values = []
    blank_values = []

    # Act & Assert: Expect ValueError due to no data points
    with pytest.raises(ValueError) as exc_info:
        CubicSplineInterpolator(input_data, standard_values, blank_values)

    # Extract the actual error message
    actual_error_message = str(exc_info.value)

    # Verify the error message matches expected text
    assert actual_error_message == ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]

# Test case for only one data point, expecting ValueError due to insufficient data
def test_single_data_point():
    # Arrange: Only one data point
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [0.5], 'coordinates': 'A1'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to insufficient data points
    with pytest.raises(ValueError) as exc_info:
        CubicSplineInterpolator(input_data, standard_values, blank_values)

    # Extract the actual error message
    actual_error_message = str(exc_info.value)

    # Verify the error message matches expected text
    assert actual_error_message == ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]


# Test case for sufficient data points, expecting successful model creation with parameters
def test_sufficient_data_points():
    # Arrange: Input with at least 3 data points
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [2], 'coordinates': 'A1'},
        {'identifier': 'S2', 'x': [2], 'y': [4], 'coordinates': 'A2'},
        {'identifier': 'S3', 'x': [3], 'y': [6], 'coordinates': 'A3'},
        {'identifier': 'S4', 'x': [4], 'y': [8], 'coordinates': 'A4'}
    ]
    standard_values = input_data
    blank_values = []

    # Act: Execute Cubic Spline Interpolation
    cs_interpolator = CubicSplineInterpolator(input_data, standard_values, blank_values)
    result = cs_interpolator.get_coefficients()

    # Assert: Check if the output contains expected keys and valid coefficients
    assert result is not None
    assert result["method"] == "cubic_spline"
    assert "coefficients" in result
    assert "metrics" in result
    assert "Additional_Table_Details" in result

    # Ensure coefficients are calculated correctly
    assert len(result["coefficients"]) == 3  # For 4 data points, we have 3 spline segments

    for coeff in result["coefficients"]:
        # Check for 'a', 'b', 'c', 'd' keys (no underscore)
        assert 'a' in coeff
        assert 'b' in coeff
        assert 'c' in coeff
        assert 'd' in coeff
