import pytest
from calculation_methods.py_calculations.five_pl_module import FiveParameterLogisticRegression
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error

# Mock handle_error to capture errors during testing
def mock_handle_error(message):
    raise ValueError(message)

# Patch the handle_error function for the duration of the tests
@pytest.fixture(autouse=True)
def patch_handle_error(monkeypatch):
    monkeypatch.setattr('calculation_methods.py_calculations.calculation_utils.handle_error', mock_handle_error)

# Test case for input with less than five data points, expecting ValueError due to insufficient data
def test_insufficient_data_points_less_than_Five():
    # Arrange: Input with less than 5 data points
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [0.5], 'coordinates': 'A1'},
        {'identifier': 'S2', 'x': [2], 'y': [0.7], 'coordinates': 'A2'},
        {'identifier': 'S3', 'x': [7], 'y': [0.8], 'coordinates': 'A3'},
        {'identifier': 'S4', 'x': [3], 'y': [0.9], 'coordinates': 'A4'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to insufficient data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        FiveParameterLogisticRegression(input_data, standard_values, blank_values)

# Test case for completely empty input data, expecting ValueError due to no data points
def test_no_data_points():
    # Arrange: Completely empty input data
    input_data = []
    standard_values = []
    blank_values = []

    # Act & Assert: Expect ValueError due to no data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        FiveParameterLogisticRegression(input_data, standard_values, blank_values)

# Test case for only one data point, expecting ValueError due to insufficient data
def test_single_data_point():
    # Arrange: Only one data point
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [0.5], 'coordinates': 'A1'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to insufficient data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        FiveParameterLogisticRegression(input_data, standard_values, blank_values)

# Test case for sufficient data points, expecting successful model creation with parameters
def test_sufficient_data():
    # Arrange: Sufficient data with multiple standard and blank values
    input_data = [
            {"coordinates": "A1", "identifier": "S5", "x": [100], "y": [640]},
            {"coordinates": "A2", "identifier": "S4", "x": [50], "y": [426]},
            {"coordinates": "A3", "identifier": "S3", "x": [25], "y": [256]},
            {"coordinates": "A4", "identifier": "S2", "x": [12.5], "y": [130]},
            {"coordinates": "A5", "identifier": "S1", "x": [6.25], "y": [70]},
            {"coordinates": "B1", "identifier": "S5", "x": [100], "y": [664]},
            {"coordinates": "B2", "identifier": "S4", "x": [50], "y": [400]},
            {"coordinates": "B3", "identifier": "S3", "x": [25], "y": [240]},
            {"coordinates": "B4", "identifier": "S2", "x": [12.5], "y": [134]},
            {"coordinates": "B5", "identifier": "S1", "x": [6.25], "y": [76]},
            {"coordinates": "C1", "identifier": "S5", "x": [100], "y": [635]},
            {"coordinates": "C2", "identifier": "S4", "x": [50], "y": [410]},
            {"coordinates": "C3", "identifier": "S3", "x": [25], "y": [244]},
            {"coordinates": "C4", "identifier": "S2", "x": [12.5], "y": [136]},
            {"coordinates": "C5", "identifier": "S1", "x": [6.25], "y": [73]},
            {"coordinates": "D1", "identifier": "U1", "x": [], "y": [400]},
            {"coordinates": "D2", "identifier": "U1", "x": [], "y": [421]},
            {"coordinates": "D3", "identifier": "U2", "x": [], "y": [301]},
            {"coordinates": "D4", "identifier": "U2", "x": [], "y": [290]},
            {"coordinates": "A6", "identifier": "B", "x": [], "y": [8]},
            {"coordinates": "B6", "identifier": "B", "x": [], "y": [10]},
            {"coordinates": "C6", "identifier": "B", "x": [], "y": [8]},
            {"coordinates": "D5", "identifier": "X", "x": [], "y": [9]},
            {"coordinates": "D6", "identifier": "B", "x": [], "y": [9]}
    ]

    standard_values = [
            {"coordinates": "A1", "identifier": "S5", "x": [100], "y": [640]},
            {"coordinates": "A2", "identifier": "S4", "x": [50], "y": [426]},
            {"coordinates": "A3", "identifier": "S3", "x": [25], "y": [256]},
            {"coordinates": "A4", "identifier": "S2", "x": [12.5], "y": [130]},
            {"coordinates": "A5", "identifier": "S1", "x": [6.25], "y": [70]},
            {"coordinates": "B1", "identifier": "S5", "x": [100], "y": [664]},
            {"coordinates": "B2", "identifier": "S4", "x": [50], "y": [400]},
            {"coordinates": "B3", "identifier": "S3", "x": [25], "y": [240]},
            {"coordinates": "B4", "identifier": "S2", "x": [12.5], "y": [134]},
            {"coordinates": "B5", "identifier": "S1", "x": [6.25], "y": [76]},
            {"coordinates": "C1", "identifier": "S5", "x": [100], "y": [635]},
            {"coordinates": "C2", "identifier": "S4", "x": [50], "y": [410]},
            {"coordinates": "C3", "identifier": "S3", "x": [25], "y": [244]},
            {"coordinates": "C4", "identifier": "S2", "x": [12.5], "y": [136]},
            {"coordinates": "C5", "identifier": "S1", "x": [6.25], "y": [73]}
    ]

    blank_values = [
            {"coordinates": "A6", "identifier": "B", "x": [], "y": [8]},
            {"coordinates": "B6", "identifier": "B", "x": [], "y": [10]},
            {"coordinates": "C6", "identifier": "B", "x": [], "y": [8]},
            {"coordinates": "D5", "identifier": "X", "x": [], "y": [9]},
            {"coordinates": "D6", "identifier": "B", "x": [], "y": [9]}
    ]

    # Act: Initialize the model with sufficient data
    model = FiveParameterLogisticRegression(input_data, standard_values, blank_values)

    # Assert: Validate the model was created successfully and parameters are present
    assert model.params is not None
    assert len(model.params) == 5
