import pytest
from calculation_methods.py_calculations.calculation_constants.constants import LINEARMODEL
from calculation_methods.py_calculations.linear_regression import LinearRegression
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error

# Mock handle_error to capture errors during testing
def mock_handle_error(message):
    raise ValueError(message)

# Patch the handle_error function for the duration of the tests
@pytest.fixture(autouse=True)
def patch_handle_error(monkeypatch):
    monkeypatch.setattr('calculation_methods.py_calculations.calculation_utils.handle_error', mock_handle_error)

# Test case for input with less than two data points, expecting ValueError due to insufficient data
def test_insufficient_data_points_less_than_two():
    # Arrange: Input with less than 2 data points
    input_data = [
        {'identifier': 'S1', 'x': [1], 'y': [0.5], 'coordinates': 'A1'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to insufficient data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        LinearRegression(input_data, standard_values, blank_values).execute()

# Test case for completely empty input data, expecting ValueError due to no data points
def test_no_data_points():
    # Arrange: Completely empty input data
    input_data = []
    standard_values = []
    blank_values = []

    # Act & Assert: Expect ValueError due to no data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        LinearRegression(input_data, standard_values, blank_values).execute()

def test_data_with_nan_or_inf():
    # Arrange: Data with NaN and Inf values
    input_data = [
        {'identifier': 'S1', 'x': [1, float('nan')], 'y': [0.5, 0.7], 'coordinates': 'A1'},
        {'identifier': 'S2', 'x': [2, float('inf')], 'y': [0.8, 0.9], 'coordinates': 'A2'}
    ]
    standard_values = input_data
    blank_values = []

    # Act & Assert: Expect ValueError due to invalid data points
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INVALID_DATA["message"]):
        LinearRegression(input_data, standard_values, blank_values).execute()


# Test case for sufficient data points, expecting successful model creation with parameters
def test_sufficient_data_points():
    # Arrange: Input with sufficient data points
    input_data = [
        {'identifier': 'S1', 'x': [0.78], 'y': [223], 'coordinates': 'A1'},
        {'identifier': 'S2', 'x': [3.4], 'y': [648], 'coordinates': 'A2'}
    ]
    standard_values = input_data
    blank_values = []
    weighted = False
    force_to_zero = True
    # Act: Execute Linear Regression
    lr = LinearRegression(input_data, standard_values, blank_values,weighted, force_to_zero)
    result = lr.execute()

    # Assert: Check if the output contains expected keys
    assert result is not None
    assert result["method"] == LINEARMODEL
    assert "function" in result
    assert "metrics" in result
    assert "Additional_Table_Details" in result
    
    # Check if the calculated slope (b) is correct (approximately)
    assert pytest.approx(result["function"]["b"], 0.01) == 195.35
    # Check if the intercept (a) is close to 0 since it's forced to zero
    assert result["function"]["a"] == 0