import json
from django.test import Client
from numpy import NaN
import pytest
import os
from calculation.calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation.calculation_methods.py_calculations.linear_regression import LinearRegression
from calculation.py_tests.unit_test_utils import Utils 

@pytest.fixture
def sample_data():
    # Use Utils.generate_sample_data() to generate the sample data dynamically.
    return Utils.generate_sample_data()

def test_linear_regression_execution(sample_data):
    # Arrange
    linear_regression = LinearRegression(
        input_data=sample_data["input_data"],
        standard_values=sample_data["standard_values"],
        blank_values=sample_data["blank_values"],
        weighted=False,
        force_to_zero=True
    )
    
    # Act
    result = linear_regression.execute()

    # Assert
    # Verify that a and b values are returned
    assert "function" in result
    assert "a" in result["function"]
    assert "b" in result["function"]

    # Verify that metrics are computed
    assert "metrics" in result
    assert "RSS" in result["metrics"]
    assert "R_Squared" in result["metrics"]
    assert "Adjusted_R_Squared" in result["metrics"]
    assert "LLD" in result["metrics"]

def test_linear_regression_coefficients(sample_data):
    # Arrange
    linear_regression = LinearRegression(
        input_data=sample_data["input_data"],
        standard_values=sample_data["standard_values"],
        blank_values=sample_data["blank_values"],
        weighted=False,
        force_to_zero=True
    )
    # Act
    linear_regression.execute()

    # Assert
    # Check the values of a and b
    assert linear_regression.a == 0  # Since force_to_zero is True
    assert linear_regression.b != 0  # b should be a valid slope value

def test_linear_regression_metrics(sample_data):
    # Arrange
    linear_regression = LinearRegression(
        input_data=sample_data["input_data"],
        standard_values=sample_data["standard_values"],
        blank_values=sample_data["blank_values"],
        weighted=False,
        force_to_zero=True
    )
    
    # Act
    result = linear_regression.execute()

    # Assert
    # Verify RSS, R_Squared, and Adjusted_R_Squared calculations
    metrics = result["metrics"]
    assert isinstance(metrics["RSS"], (int, float))
    assert isinstance(metrics["R_Squared"], (int, float))
    assert isinstance(metrics["Adjusted_R_Squared"], (int, float))
    assert isinstance(metrics["LLD"], (int, float, type(None)))  # LLD might be None

def test_linear_regression_table_data(sample_data):
    # Arrange
    linear_regression = LinearRegression(
        input_data=sample_data["input_data"],
        standard_values=sample_data["standard_values"],
        blank_values=sample_data["blank_values"],
        weighted=False,
        force_to_zero=True
    )
    
    # Act
    result = linear_regression.execute()

    # Assert
    # Verify the table data
    table_data = result["Additional_Table_Details"]
    assert isinstance(table_data, list)
    
    # Get unique identifiers from input data (S1-S5, U1, U2 from Utils)
    unique_identifiers = len(set(item['identifier'] for item in sample_data["input_data"]))
    assert len(table_data) == unique_identifiers

def test_linear_regression_invalid_data():
    # Arrange
    invalid_data = [
        {"coordinates": "A1", "identifier": "S5", "x": [NaN], "y": [640]}
    ]
    
    # Act
    lr = LinearRegression(input_data=invalid_data, standard_values=[], blank_values=[], weighted=False, force_to_zero=True)

    # Assert
    with pytest.raises(ValueError, match=ErrorMessages.ERROR_INSUFFICIENT_DATA_POINTS["message"]):
        lr.execute()