import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error

class TestCalculateCV:

    # Tests the calculation of the coefficient of variation (CV) with a typical set of positive values.
    @pytest.mark.happy_path
    def test_calculate_cv_typical_values(self):
        # Arrange: Use a list of diverse positive numbers to represent a common case.
        values = [10, 20, 30, 40, 50]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, which is (standard deviation / mean) * 100.
        result = stats.calculate_statistics()
        # Compute expected CV using NumPy for comparison.
        expected_cv = (np.std(values) / np.mean(values)) * 100
        # Assert:
        assert result["cv"] == pytest.approx(expected_cv, rel=1e-2)

    # Tests CV calculation when the input includes NaN values, ensuring proper handling with NaN-aware functions.
    @pytest.mark.happy_path
    def test_calculate_cv_with_nan_values(self):
        # Arrange: Include a NaN in a list of typical values to simulate missing data.
        values = [10, 20, np.nan, 40, 50]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, expecting NaN to be ignored in the computation.
        result = stats.calculate_statistics()
        # Compute expected CV using NaN-safe NumPy functions (nanstd, nanmean).
        expected_cv = (np.nanstd(values) / np.nanmean(values)) * 100
        # Assert: Check that the result approximates the expected CV, accounting for NaN handling.
        assert result["cv"] == pytest.approx(expected_cv, rel=1e-2)

    # Tests CV with a single value, where variation is undefined, expecting a result of 0.
    @pytest.mark.edge_case
    def test_calculate_cv_single_value(self):
        # Arrange: Use a single value to test the edge case of minimal data.
        values = [10]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, where standard deviation is 0 due to one value.
        result = stats.calculate_statistics()
        # Assert: Verify the result is 0, as CV is zero when there’s no variation.
        assert result["cv"] == 0.0  # CV of a single value should be 0

    # Tests CV with identical values, expecting no variation and thus a CV of 0.
    @pytest.mark.edge_case
    def test_calculate_cv_identical_values(self):
        # Arrange: Use a list of identical values to test zero-variation case.
        values = [10, 10, 10, 10, 10]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, where standard deviation is 0 due to identical values.
        result = stats.calculate_statistics()
        # Assert: Confirm the result is 0, indicating no variability among identical values.
        assert result["cv"] == 0.0  # CV of identical values should be 0

    # Tests CV with an empty list, expecting a None result due to insufficient data.
    @pytest.mark.edge_case
    def test_calculate_cv_empty_list(self):
        # Arrange: Use an empty list to test the edge case of no data.
        values = []
        # Act & Assert: Ensure a ValueError is raised for empty input
        with pytest.raises(ValueError, match=ErrorMessages.ERROR_MISSING_INPUT_DATA["message"]):
            stats = SampleTypeStatistics(input_values=values)

    # Tests CV with values summing to a zero mean, expecting None due to division by zero.
    @pytest.mark.edge_case
    def test_calculate_cv_zero_mean(self):
        # Arrange: Use values that average to 0 to test division-by-zero scenario in CV calculation.
        values = [-10, 0, 10]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, where mean is 0, making CV undefined (std / 0).
        result = stats.calculate_statistics()
        # Assert: Confirm the result is None, as CV cannot be computed with a zero mean.
        assert result["cv"] == None

    # Tests CV with all NaN values, expecting a NaN result due to undefined statistics.
    @pytest.mark.edge_case
    def test_calculate_cv_all_nan(self):
        # Arrange: Use a list of all NaN values to test the edge case of completely missing data.
        values = [np.nan, np.nan, np.nan]
        stats = SampleTypeStatistics(input_values=values, metrics=["cv"])
        # Act: Calculate CV, where both mean and std are NaN due to all NaN inputs.
        result = stats.calculate_statistics()
        # Assert: Verify the result is NaN, reflecting undefined statistics with all NaN values.
        assert np.isnan(result["cv"])  # CV should be NaN if all values are NaN