import pytest
import numpy as np
import random
from calculation.calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from unittest.mock import patch


class TestCalculateMean:

    @pytest.mark.happy_path
    def test_calculate_mean_typical_values(self):
        #Test calculate_mean with a typical list of numbers.
        #Arrange
        input_values = [random.randint(1, 100) for _ in range(5)]
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Assert
        assert result["mean"] == np.mean(input_values)

    @pytest.mark.happy_path
    def test_calculate_mean_with_nan(self):
        #Test calculate_mean with a list containing NaN values.
        #Arrange
        input_values = [random.randint(1, 100) for _ in range(4)] + [np.nan]
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Act
        assert result["mean"] == np.nanmean(input_values)

    @pytest.mark.happy_path
    def test_calculate_mean_single_value(self):
        #Test calculate_mean with a single value.
        input_values = [random.randint(1, 100)]
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Assert
        assert result["mean"] == input_values[0]

    @pytest.mark.edge_case
    def test_calculate_mean_all_nan(self):
        #Test calculate_mean with a list of all NaN values.
        input_values = [np.nan] * 3
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Assert
        assert np.isnan(result["mean"])

    @pytest.mark.edge_case
    def test_calculate_mean_large_numbers(self):
        #Test calculate_mean with very large numbers.
        #Arrange
        large_number = random.uniform(1e18, 1e19)
        input_values = [large_number] * 3
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Assert
        assert result["mean"] == pytest.approx(large_number)

    @pytest.mark.edge_case
    def test_calculate_mean_mixed_types(self):
        #Test calculate_mean with mixed types (integers and floats).
        
        #Arrange
        input_values = [random.randint(1, 50), random.uniform(1.0, 50.0), random.randint(1, 50), random.uniform(1.0, 50.0)]
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])
        
        #Act
        result = stats.calculate_statistics()
        
        #Assert
        assert result["mean"] == np.mean(input_values)
