import pytest
import numpy as np
import random
from calculation.calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.calculation_utils import handle_error
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

class TestCalculateMin:
    
    @pytest.mark.happy_path
    def test_calculate_min_normal_values(self):
        # Test calculate_min with a normal list of positive numbers.

        # Arrange
        values = random.sample(range(1, 100), 5)
        expected_min = min(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min

    @pytest.mark.happy_path
    def test_calculate_min_with_negative_values(self):
        # Test calculate_min with a list containing negative numbers.

        # Arrange
        values = random.sample(range(-100, 100), 5)
        expected_min = min(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min

    @pytest.mark.happy_path
    def test_calculate_min_with_floats(self):
        # Test calculate_min with a list of floating-point numbers.

        # Arrange
        values = [random.uniform(-50.5, 50.5) for _ in range(5)]
        expected_min = min(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min
    @pytest.mark.edge_case
    def test_calculate_min_with_nan_values(self):
        # Test calculate_min with a list containing NaN values.

        # Arrange
        values = [np.nan, random.randint(1, 50), random.randint(51, 100), np.nan, random.randint(101, 200)]
        expected_min = np.nanmin(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min

    @pytest.mark.edge_case
    def test_calculate_min_with_identical_values(self):
        # Test calculate_min with a list of identical values.

        # Arrange
        value = random.randint(1, 100)
        values = [value] * 5
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == value

    @pytest.mark.edge_case
    def test_calculate_min_with_large_numbers(self):
        # Test calculate_min with very large numbers.

        # Arrange
        values = [random.uniform(1e10, 1e12) for _ in range(3)]
        expected_min = min(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min

    @pytest.mark.edge_case
    def test_calculate_min_with_infinities(self):
        # Test calculate_min with positive and negative infinity.

        # Arrange
        values = [np.inf, -np.inf, random.randint(-50, 50), random.randint(1, 100)]
        expected_min = min(values)
        stats = SampleTypeStatistics(input_values=values, metrics=['min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert result["min"] == expected_min
