import pytest
import numpy as np
import random
from calculation.calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages

class TestCalculateStandardDeviation:

    @pytest.mark.happy_path
    def test_standard_deviation_typical_case(self):
        #Test standard deviation with a typical set of positive numbers.
        # Arrange
        values = [random.randint(1, 100) for _ in range(5)]
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = np.std(values, ddof=0)
        assert result['std'] == expected_std

    @pytest.mark.happy_path
    def test_standard_deviation_with_negative_numbers(self):
        #Test standard deviation with a set of negative numbers.
        # Arrange
        values = [random.randint(-100, -1) for _ in range(5)]
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = np.std(values, ddof=0)
        assert result['std'] == expected_std

    @pytest.mark.happy_path
    def test_standard_deviation_with_mixed_numbers(self):
        #Test standard deviation with a mix of positive and negative numbers.
        # Arrange
        values = [random.randint(-50, 50) for _ in range(5)]
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = np.std(values, ddof=0)
        assert result['std'] == expected_std

    @pytest.mark.edge_case
    def test_standard_deviation_single_value(self):
        #Test standard deviation with a single value.
        # Arrange
        value = random.randint(1, 100)
        values = [value]
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = 0.0  # Standard deviation of a single value is 0
        assert result['std'] == expected_std

    @pytest.mark.edge_case
    def test_standard_deviation_identical_values(self):
        #Test standard deviation with identical values.
        # Arrange
        value = random.randint(1, 100)
        values = [value] * 5  # Identical values
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = 0.0  # Standard deviation of identical values is 0
        assert result['std'] == expected_std

    @pytest.mark.edge_case
    def test_standard_deviation_with_nan_values(self):
        #Test standard deviation with NaN values in the input.
        # Arrange
        values = [random.randint(1, 100) for _ in range(3)] + [np.nan] + [random.randint(1, 100)]
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        expected_std = np.nanstd(values)
        assert result['std'] == expected_std

    @pytest.mark.edge_case
    def test_standard_deviation_all_nan_values(self):
        #Test standard deviation with all NaN values.
        # Arrange
        values = [np.nan] * 5  # All NaN values
        stats = SampleTypeStatistics(input_values=values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert np.isnan(result['std'])

