import pytest
import numpy as np
from calculation.calculation_methods.py_calculations.sample_statistics import SampleTypeStatistics
from calculation_methods.py_calculations.error_handler.error_messages import ErrorMessages
from calculation_methods.py_calculations.calculation_utils import handle_error

# Mock the handle_error function to raise exceptions for testing
def mock_handle_error(message):
    raise ValueError(message)

# Apply the mock
handle_error = mock_handle_error

# Set a random seed for reproducibility
np.random.seed(42)

class TestSampleTypeStatistics:
    @pytest.mark.happy_path
    # Tests if calculate_statistics correctly computes the mean for a small, random dataset.
    def test_calculate_statistics_mean_random(self):
        # Arrange
        input_values = np.random.uniform(low=1, high=100, size=10).tolist()
        expected_mean = np.mean(input_values)
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert np.isclose(result['mean'], expected_mean)

    @pytest.mark.happy_path
    # Verifies that calculate_statistics accurately calculates the standard deviation for a small, random dataset.
    def test_calculate_statistics_std_random(self):
        # Arrange
        input_values = np.random.normal(loc=50, scale=15, size=10).tolist()
        expected_std = np.std(input_values, ddof=0)
        stats = SampleTypeStatistics(input_values=input_values, metrics=['std'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert np.isclose(result['std'], expected_std)

    @pytest.mark.edge_case
    # Checks if calculate_statistics handles a large dataset (1000 values) correctly across multiple metrics.
    def test_calculate_statistics_large_random_dataset(self):
        # Arrange
        input_values = np.random.uniform(low=0, high=1000, size=1000).tolist()
        expected_mean = np.mean(input_values)
        expected_std = np.std(input_values, ddof=0)
        expected_sum = np.sum(input_values)
        expected_max = max(input_values)
        expected_min = min(input_values)
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean', 'std', 'sum', 'max', 'min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert np.isclose(result['mean'], expected_mean)
        assert np.isclose(result['std'], expected_std)
        assert np.isclose(result['sum'], expected_sum)
        assert result['max'] == expected_max
        assert result['min'] == expected_min

    @pytest.mark.edge_case
    # Ensures calculate_statistics properly ignores NaN values and computes metrics on valid data only.
    def test_calculate_statistics_with_some_nan_values_random(self):
        # Arrange
        input_values = np.random.uniform(low=10, high=50, size=10).tolist()
        input_values[2] = np.nan  # Introduce NaN into the dataset
        input_values[5] = np.nan  # Another NaN
        cleaned_values = [x for x in input_values if not np.isnan(x)]  # Remove NaNs for expected values
        expected_mean = np.mean(cleaned_values)
        expected_std = np.std(cleaned_values, ddof=0)
        expected_sum = np.sum(cleaned_values)
        expected_max = max(cleaned_values)
        expected_min = min(cleaned_values)
        stats = SampleTypeStatistics(input_values=input_values, metrics=['mean', 'std', 'sum', 'max', 'min'])

        # Act
        result = stats.calculate_statistics()

        # Assert
        assert np.isclose(result['mean'], expected_mean)
        assert np.isclose(result['std'], expected_std)
        assert np.isclose(result['sum'], expected_sum)
        assert result['max'] == expected_max
        assert result['min'] == expected_min